#define STB_IMAGE_IMPLEMENTATION
#include "stb_image.h" // TODO: Probably switch to SDL_image

//#define _USE_MATH_DEFINES // Will be needed when/if I need to # include <cmath>

#define GLM_FORCE_RADIANS
#define GLM_FORCE_DEPTH_ZERO_TO_ONE

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>

#include <iostream>
#include <fstream>
#include <algorithm>
#include <vector>
#include <array>
#include <set>
#include <optional>
#include <chrono>

#include "consts.hpp"
#include "utils.hpp"

#include "game-gui-sdl.hpp"

using namespace std;
using namespace glm;

const int SCREEN_WIDTH = 800;
const int SCREEN_HEIGHT = 600;

const int MAX_FRAMES_IN_FLIGHT = 2;

/*** START OF REFACTORED CODE ***/
#ifdef NDEBUG
   const bool enableValidationLayers = false;
#else
   const bool enableValidationLayers = true;
#endif

const vector<const char*> validationLayers = {
   "VK_LAYER_KHRONOS_validation"
};

const vector<const char*> deviceExtensions = {
   VK_KHR_SWAPCHAIN_EXTENSION_NAME
};

struct QueueFamilyIndices {
    optional<uint32_t> graphicsFamily;
    optional<uint32_t> presentFamily;

    bool isComplete() {
        return graphicsFamily.has_value() && presentFamily.has_value();
    }
};

struct SwapChainSupportDetails {
    VkSurfaceCapabilitiesKHR capabilities;
    vector<VkSurfaceFormatKHR> formats;
    vector<VkPresentModeKHR> presentModes;
};
/*** END OF REFACTORED CODE ***/

struct Vertex {
   glm::vec3 pos;
   glm::vec3 color;
   glm::vec2 texCoord;
};

struct OverlayVertex {
   glm::vec3 pos;
   glm::vec2 texCoord;
};

struct UniformBufferObject {
   alignas(16) mat4 model;
   alignas(16) mat4 view;
   alignas(16) mat4 proj;
};

struct DescriptorInfo {
   VkDescriptorType type;
   VkShaderStageFlags stageFlags;

   vector<VkDescriptorBufferInfo>* bufferDataList;
   VkDescriptorImageInfo* imageData;
};

struct GraphicsPipelineInfo {
   VkPipelineLayout pipelineLayout;
   VkPipeline pipeline;

   VkVertexInputBindingDescription bindingDescription;
   vector<VkVertexInputAttributeDescription> attributeDescriptions;

   vector<DescriptorInfo> descriptorInfoList;

   VkDescriptorPool descriptorPool;
   VkDescriptorSetLayout descriptorSetLayout;
   vector<VkDescriptorSet> descriptorSets;

   size_t numVertices; // Currently unused
   VkBuffer vertexBuffer;
   VkDeviceMemory vertexBufferMemory;

   size_t numIndices;
   VkBuffer indexBuffer;
   VkDeviceMemory indexBufferMemory;
};

/*** START OF REFACTORED CODE ***/
VkResult CreateDebugUtilsMessengerEXT(VkInstance instance,
      const VkDebugUtilsMessengerCreateInfoEXT* pCreateInfo,
      const VkAllocationCallbacks* pAllocator,
      VkDebugUtilsMessengerEXT* pDebugMessenger) {
   auto func = (PFN_vkCreateDebugUtilsMessengerEXT) vkGetInstanceProcAddr(instance, "vkCreateDebugUtilsMessengerEXT");

   if (func != nullptr) {
      return func(instance, pCreateInfo, pAllocator, pDebugMessenger);
   } else {
      return VK_ERROR_EXTENSION_NOT_PRESENT;
   }
}

void DestroyDebugUtilsMessengerEXT(VkInstance instance,
      VkDebugUtilsMessengerEXT debugMessenger,
      const VkAllocationCallbacks* pAllocator) {
   auto func = (PFN_vkDestroyDebugUtilsMessengerEXT) vkGetInstanceProcAddr(instance, "vkDestroyDebugUtilsMessengerEXT");

   if (func != nullptr) {
      func(instance, debugMessenger, pAllocator);
   }
}

class VulkanGame {
   public:
      void run() {
         if (initWindow() == RTWO_ERROR) {
            return;
         }
         initVulkan();
         mainLoop();
         cleanup();
      }

   private:
      GameGui* gui = new GameGui_SDL();

      SDL_version sdlVersion;
      SDL_Window* window = nullptr;
      SDL_Renderer* gRenderer = nullptr;
/*** END OF REFACTORED CODE ***/
      SDL_Texture* uiOverlay = nullptr;

      TTF_Font* gFont = nullptr;
      SDL_Texture* uiText = nullptr;
      SDL_Texture* uiImage = nullptr;

/*** START OF REFACTORED CODE ***/
      VkInstance instance;
      VkDebugUtilsMessengerEXT debugMessenger;
      VkSurfaceKHR surface;

      VkPhysicalDevice physicalDevice = VK_NULL_HANDLE;
      VkDevice device;

      VkQueue graphicsQueue;
      VkQueue presentQueue;

      VkSwapchainKHR swapChain;
      vector<VkImage> swapChainImages;
      VkFormat swapChainImageFormat;
      VkExtent2D swapChainExtent;
      vector<VkImageView> swapChainImageViews;
/*** END OF REFACTORED CODE ***/
      vector<VkFramebuffer> swapChainFramebuffers;

      VkRenderPass renderPass;

      VkCommandPool commandPool;
      vector<VkCommandBuffer> commandBuffers;

      // The images and the sampler are used to store data for specific attributes. I probably
      // want to keep them separate from the GraphicsPipelineInfo objects and start passing
      // references to them once I start defining uniform and varying attributes in GraphicsPipelineInfo objects

      VkImage depthImage;
      VkDeviceMemory depthImageMemory;
      VkImageView depthImageView;

      VkImage textureImage;
      VkDeviceMemory textureImageMemory;
      VkImageView textureImageView;

      VkImage overlayImage;
      VkDeviceMemory overlayImageMemory;
      VkImageView overlayImageView;

      VkImage sdlOverlayImage;
      VkDeviceMemory sdlOverlayImageMemory;
      VkImageView sdlOverlayImageView;

      VkSampler textureSampler;

      // These are currently to store the MVP matrix
      // I should figure out if it makes sense to use them for other uniforms in the future
      // If not, I should rename them to better indicate their puprose.
      // I should also decide if I can use these for all shaders, or if I need a separapte set of buffers for each one
      vector<VkBuffer> uniformBuffers;
      vector<VkDeviceMemory> uniformBuffersMemory;

      VkDescriptorImageInfo sceneImageInfo;
      VkDescriptorImageInfo overlayImageInfo;

      vector<VkDescriptorBufferInfo> uniformBufferInfoList;

      GraphicsPipelineInfo scenePipeline;
      GraphicsPipelineInfo overlayPipeline;

      vector<VkSemaphore> imageAvailableSemaphores;
      vector<VkSemaphore> renderFinishedSemaphores;
      vector<VkFence> inFlightFences;

      size_t currentFrame = 0;

      bool framebufferResized = false;

/*** START OF REFACTORED CODE ***/
      bool initWindow() {
         if (gui->init() == RTWO_ERROR) {
            cout << "UI library could not be initialized!" << endl;
            cout << SDL_GetError() << endl;
            return RTWO_ERROR;
         }
         cout << "GUI init succeeded" << endl;

         window = (SDL_Window*) gui->createWindow("Vulkan Game", SCREEN_WIDTH, SCREEN_HEIGHT, true);
         if (window == nullptr) {
            cout << "Window could not be created!" << endl;
            return RTWO_ERROR;
         }

         gRenderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);
         if (gRenderer == nullptr) {
            cout << "Renderer could not be created! SDL Error: " << SDL_GetError() << endl;
            return RTWO_ERROR;
         }
/*** END OF REFACTORED CODE ***/

         SDL_VERSION(&sdlVersion);

         // In SDL 2.0.10 (currently, the latest), SDL_TEXTUREACCESS_TARGET is required to get a transparent overlay working
         // However, the latest SDL version available through homebrew on Mac is 2.0.9, which requires SDL_TEXTUREACCESS_STREAMING
         // I tried building sdl 2.0.10 (and sdl_image and sdl_ttf) from source on Mac, but had some issues, so this is easier
         // until the homebrew recipe is updated
         if (sdlVersion.major == 2 && sdlVersion.minor == 0 && sdlVersion.patch == 9) {
            uiOverlay = SDL_CreateTexture(gRenderer, SDL_PIXELFORMAT_RGBA8888, SDL_TEXTUREACCESS_STREAMING, SCREEN_WIDTH, SCREEN_HEIGHT);
         } else {
            uiOverlay = SDL_CreateTexture(gRenderer, SDL_PIXELFORMAT_RGBA8888, SDL_TEXTUREACCESS_TARGET, SCREEN_WIDTH, SCREEN_HEIGHT);
         }

         if (uiOverlay == nullptr) {
            cout << "Unable to create blank texture! SDL Error: " << SDL_GetError() << endl;
         }
         if (SDL_SetTextureBlendMode(uiOverlay, SDL_BLENDMODE_BLEND) != 0) {
            cout << "Unable to set texture blend mode! SDL Error: " << SDL_GetError() << endl;
         }

         gFont = TTF_OpenFont("fonts/lazy.ttf", 28);
         if (gFont == nullptr) {
            cout << "Failed to load lazy font! SDL_ttf Error: " << TTF_GetError() << endl;
            return RTWO_ERROR;
         }

         SDL_Color textColor = { 0, 0, 0 };

         SDL_Surface* textSurface = TTF_RenderText_Solid(gFont, "Great sucess!", textColor);
         if (textSurface == nullptr) {
            cout << "Unable to render text surface! SDL_ttf Error: " << TTF_GetError() << endl;
            return RTWO_ERROR;
         }

         uiText = SDL_CreateTextureFromSurface(gRenderer, textSurface);
         if (uiText == nullptr) {
            cout << "Unable to create texture from rendered text! SDL Error: " << SDL_GetError() << endl;
            SDL_FreeSurface(textSurface);
            return RTWO_ERROR;
         }

         SDL_FreeSurface(textSurface);

         // TODO: Load a PNG instead
         SDL_Surface* uiImageSurface = SDL_LoadBMP("assets/images/spaceship.bmp");
         if (uiImageSurface == nullptr) {
            cout << "Unable to load image " << "spaceship.bmp" << "! SDL Error: " << SDL_GetError() << endl;
            return RTWO_ERROR;
         }

         uiImage = SDL_CreateTextureFromSurface(gRenderer, uiImageSurface);
         if (uiImage == nullptr) {
            cout << "Unable to create texture from BMP surface! SDL Error: " << SDL_GetError() << endl;
            SDL_FreeSurface(uiImageSurface);
            return RTWO_ERROR;
         }

         SDL_FreeSurface(uiImageSurface);

         return RTWO_SUCCESS;
      }

/*** START OF REFACTORED CODE ***/
      void initVulkan() {
         createInstance();
         setupDebugMessenger();
         createSurface();
         pickPhysicalDevice();
         createLogicalDevice();
         createSwapChain();
         createImageViews();
/*** END OF REFACTORED CODE ***/
         createRenderPass();

         createCommandPool();

         createImageResources("textures/texture.jpg", textureImage, textureImageMemory, textureImageView);
         createImageResourcesFromSDLTexture(uiOverlay, sdlOverlayImage, sdlOverlayImageMemory, sdlOverlayImageView);
         createTextureSampler();

         sceneImageInfo = {};
         sceneImageInfo.imageLayout = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL;
         sceneImageInfo.imageView = textureImageView;
         sceneImageInfo.sampler = textureSampler;

         overlayImageInfo = {};
         overlayImageInfo.imageLayout = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL;
         overlayImageInfo.imageView = sdlOverlayImageView;
         overlayImageInfo.sampler = textureSampler;

         vector<Vertex> sceneVertices = {
            {{-0.5f, -0.5f, -0.5f}, {1.0f, 0.0f, 0.0f}, {0.0f, 1.0f}},
            {{ 0.5f, -0.5f, -0.5f}, {0.0f, 1.0f, 0.0f}, {1.0f, 1.0f}},
            {{ 0.5f,  0.5f, -0.5f}, {0.0f, 0.0f, 1.0f}, {1.0f, 0.0f}},
            {{-0.5f,  0.5f, -0.5f}, {1.0f, 1.0f, 1.0f}, {0.0f, 0.0f}},

            {{-0.5f, -0.5f,  0.0f}, {1.0f, 0.0f, 0.0f}, {0.0f, 1.0f}},
            {{ 0.5f, -0.5f,  0.0f}, {0.0f, 1.0f, 0.0f}, {1.0f, 1.0f}},
            {{ 0.5f,  0.5f,  0.0f}, {0.0f, 0.0f, 1.0f}, {1.0f, 0.0f}},
            {{-0.5f,  0.5f,  0.0f}, {1.0f, 1.0f, 1.0f}, {0.0f, 0.0f}}
         };
         vector<uint16_t> sceneIndices = {
            0, 1, 2, 2, 3, 0,
            4, 5, 6, 6, 7, 4
         };

         initGraphicsPipelineInfo(scenePipeline,
            sceneVertices.data(), sizeof(Vertex), sceneVertices.size(),
            sceneIndices.data(), sizeof(uint16_t), sceneIndices.size());

         addAttributeDescription(scenePipeline, VK_FORMAT_R32G32B32_SFLOAT, offset_of(&Vertex::pos));
         addAttributeDescription(scenePipeline, VK_FORMAT_R32G32B32_SFLOAT, offset_of(&Vertex::color));
         addAttributeDescription(scenePipeline, VK_FORMAT_R32G32_SFLOAT, offset_of(&Vertex::texCoord));

         addDescriptorInfo(scenePipeline, VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER, VK_SHADER_STAGE_VERTEX_BIT, &uniformBufferInfoList, nullptr);
         addDescriptorInfo(scenePipeline, VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER, VK_SHADER_STAGE_FRAGMENT_BIT, nullptr, &sceneImageInfo);

         createDescriptorSetLayout(scenePipeline);


         vector<OverlayVertex> overlayVertices = {
            {{-1.0f,  1.0f,  0.0f}, {0.0f, 1.0f}},
            {{ 1.0f,  1.0f,  0.0f}, {1.0f, 1.0f}},
            {{ 1.0f, -1.0f,  0.0f}, {1.0f, 0.0f}},
            {{-1.0f, -1.0f,  0.0f}, {0.0f, 0.0f}}
         };
         vector<uint16_t> overlayIndices = {
            0, 1, 2, 2, 3, 0
         };

         initGraphicsPipelineInfo(overlayPipeline,
            overlayVertices.data(), sizeof(OverlayVertex), overlayVertices.size(),
            overlayIndices.data(), sizeof(uint16_t), overlayIndices.size());

         addAttributeDescription(overlayPipeline, VK_FORMAT_R32G32B32_SFLOAT, offset_of(&OverlayVertex::pos));
         addAttributeDescription(overlayPipeline, VK_FORMAT_R32G32_SFLOAT, offset_of(&OverlayVertex::texCoord));

         addDescriptorInfo(overlayPipeline, VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER, VK_SHADER_STAGE_FRAGMENT_BIT, nullptr, &overlayImageInfo);

         createDescriptorSetLayout(overlayPipeline);

         createBufferResources();

         createSyncObjects();
      }

      void createInstance() {
         if (enableValidationLayers && !checkValidationLayerSupport()) {
            throw runtime_error("validation layers requested, but not available!");
         }

         VkApplicationInfo appInfo = {};
         appInfo.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
         appInfo.pApplicationName = "Vulkan Game";
         appInfo.applicationVersion = VK_MAKE_VERSION(1, 0, 0);
         appInfo.pEngineName = "No Engine";
         appInfo.engineVersion = VK_MAKE_VERSION(1, 0, 0);
         appInfo.apiVersion = VK_API_VERSION_1_0;

         VkInstanceCreateInfo createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
         createInfo.pApplicationInfo = &appInfo;

         vector<const char*> extensions = getRequiredExtensions();
         createInfo.enabledExtensionCount = static_cast<uint32_t>(extensions.size());
         createInfo.ppEnabledExtensionNames = extensions.data();

         cout << endl << "Extensions:" << endl;
         for (const char* extensionName : extensions) {
            cout << extensionName << endl;
         }
         cout << endl;

         VkDebugUtilsMessengerCreateInfoEXT debugCreateInfo;
         if (enableValidationLayers) {
            createInfo.enabledLayerCount = static_cast<uint32_t>(validationLayers.size());
            createInfo.ppEnabledLayerNames = validationLayers.data();

            populateDebugMessengerCreateInfo(debugCreateInfo);
            createInfo.pNext = &debugCreateInfo;
         } else {
            createInfo.enabledLayerCount = 0;

            createInfo.pNext = nullptr;
         }

         if (vkCreateInstance(&createInfo, nullptr, &instance) != VK_SUCCESS) {
            throw runtime_error("failed to create instance!");
         }
      }

      bool checkValidationLayerSupport() {
         uint32_t layerCount;
         vkEnumerateInstanceLayerProperties(&layerCount, nullptr);

         vector<VkLayerProperties> availableLayers(layerCount);
         vkEnumerateInstanceLayerProperties(&layerCount, availableLayers.data());

         for (const char* layerName : validationLayers) {
            bool layerFound = false;

            for (const auto& layerProperties : availableLayers) {
               if (strcmp(layerName, layerProperties.layerName) == 0) {
                  layerFound = true;
                  break;
               }
            }

            if (!layerFound) {
               return false;
            }
         }

         return true;
      }

/*** START OF REFACTORED CODE ***/
      vector<const char*> getRequiredExtensions() {
         vector<const char*> extensions = gui->getRequiredExtensions();

         if (enableValidationLayers) {
            extensions.push_back(VK_EXT_DEBUG_UTILS_EXTENSION_NAME);
         }

         return extensions;
      }

      void setupDebugMessenger() {
         if (!enableValidationLayers) return;

         VkDebugUtilsMessengerCreateInfoEXT createInfo;
         populateDebugMessengerCreateInfo(createInfo);

         if (CreateDebugUtilsMessengerEXT(instance, &createInfo, nullptr, &debugMessenger) != VK_SUCCESS) {
            throw runtime_error("failed to set up debug messenger!");
         }
      }

      void populateDebugMessengerCreateInfo(VkDebugUtilsMessengerCreateInfoEXT& createInfo) {
         createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT;
         createInfo.messageSeverity = VK_DEBUG_UTILS_MESSAGE_SEVERITY_VERBOSE_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_SEVERITY_WARNING_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_SEVERITY_ERROR_BIT_EXT;
         createInfo.messageType = VK_DEBUG_UTILS_MESSAGE_TYPE_GENERAL_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_TYPE_VALIDATION_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_TYPE_PERFORMANCE_BIT_EXT;
         createInfo.pfnUserCallback = debugCallback;
      }

      void createSurface() {
         if (gui->createVulkanSurface(instance, &surface) == RTWO_ERROR) {
            throw runtime_error("failed to create window surface!");
         }
      }

      void pickPhysicalDevice() {
         uint32_t deviceCount = 0;
         vkEnumeratePhysicalDevices(instance, &deviceCount, nullptr);

         if (deviceCount == 0) {
            throw runtime_error("failed to find GPUs with Vulkan support!");
         }

         vector<VkPhysicalDevice> devices(deviceCount);
         vkEnumeratePhysicalDevices(instance, &deviceCount, devices.data());

         cout << endl << "Graphics cards:" << endl;
         for (const VkPhysicalDevice& device : devices) {
            if (isDeviceSuitable(device)) {
               physicalDevice = device;
               break;
            }
         }
         cout << endl;

         if (physicalDevice == VK_NULL_HANDLE) {
            throw runtime_error("failed to find a suitable GPU!");
         }
      }

      bool isDeviceSuitable(VkPhysicalDevice device) {
         VkPhysicalDeviceProperties deviceProperties;
         vkGetPhysicalDeviceProperties(device, &deviceProperties);

         cout << "Device: " << deviceProperties.deviceName << endl;

         QueueFamilyIndices indices = findQueueFamilies(device);
         bool extensionsSupported = checkDeviceExtensionSupport(device);
         bool swapChainAdequate = false;

         if (extensionsSupported) {
            SwapChainSupportDetails swapChainSupport = querySwapChainSupport(device);
            swapChainAdequate = !swapChainSupport.formats.empty() && !swapChainSupport.presentModes.empty();
         }

         VkPhysicalDeviceFeatures supportedFeatures;
         vkGetPhysicalDeviceFeatures(device, &supportedFeatures);

         return indices.isComplete() && extensionsSupported && swapChainAdequate && supportedFeatures.samplerAnisotropy;
      }

      bool checkDeviceExtensionSupport(VkPhysicalDevice device) {
         uint32_t extensionCount;
         vkEnumerateDeviceExtensionProperties(device, nullptr, &extensionCount, nullptr);

         vector<VkExtensionProperties> availableExtensions(extensionCount);
         vkEnumerateDeviceExtensionProperties(device, nullptr, &extensionCount, availableExtensions.data());

         set<string> requiredExtensions(deviceExtensions.begin(), deviceExtensions.end());

         for (const auto& extension : availableExtensions) {
            requiredExtensions.erase(extension.extensionName);
         }

         return requiredExtensions.empty();
      }

      void createLogicalDevice() {
         QueueFamilyIndices indices = findQueueFamilies(physicalDevice);

         vector<VkDeviceQueueCreateInfo> queueCreateInfos;
         set<uint32_t> uniqueQueueFamilies = {indices.graphicsFamily.value(), indices.presentFamily.value()};

         float queuePriority = 1.0f;
         for (uint32_t queueFamily : uniqueQueueFamilies) {
            VkDeviceQueueCreateInfo queueCreateInfo = {};
            queueCreateInfo.sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
            queueCreateInfo.queueFamilyIndex = queueFamily;
            queueCreateInfo.queueCount = 1;
            queueCreateInfo.pQueuePriorities = &queuePriority;

            queueCreateInfos.push_back(queueCreateInfo);
         }

         VkPhysicalDeviceFeatures deviceFeatures = {};
         deviceFeatures.samplerAnisotropy = VK_TRUE;

         VkDeviceCreateInfo createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO;
         createInfo.queueCreateInfoCount = static_cast<uint32_t>(queueCreateInfos.size());
         createInfo.pQueueCreateInfos = queueCreateInfos.data();

         createInfo.pEnabledFeatures = &deviceFeatures;

         createInfo.enabledExtensionCount = static_cast<uint32_t>(deviceExtensions.size());
         createInfo.ppEnabledExtensionNames = deviceExtensions.data();

         // These fields are ignored  by up-to-date Vulkan implementations,
         // but it's a good idea to set them for backwards compatibility
         if (enableValidationLayers) {
            createInfo.enabledLayerCount = static_cast<uint32_t>(validationLayers.size());
            createInfo.ppEnabledLayerNames = validationLayers.data();
         } else {
            createInfo.enabledLayerCount = 0;
         }

         if (vkCreateDevice(physicalDevice, &createInfo, nullptr, &device) != VK_SUCCESS) {
            throw runtime_error("failed to create logical device!");
         }

         vkGetDeviceQueue(device, indices.graphicsFamily.value(), 0, &graphicsQueue);
         vkGetDeviceQueue(device, indices.presentFamily.value(), 0, &presentQueue);
      }

      void createSwapChain() {
         SwapChainSupportDetails swapChainSupport = querySwapChainSupport(physicalDevice);

         VkSurfaceFormatKHR surfaceFormat = chooseSwapSurfaceFormat(swapChainSupport.formats);
         VkPresentModeKHR presentMode = chooseSwapPresentMode(swapChainSupport.presentModes);
         VkExtent2D extent = chooseSwapExtent(swapChainSupport.capabilities);

         uint32_t imageCount = swapChainSupport.capabilities.minImageCount + 1;
         if (swapChainSupport.capabilities.maxImageCount > 0 && imageCount > swapChainSupport.capabilities.maxImageCount) {
            imageCount = swapChainSupport.capabilities.maxImageCount;
         }

         VkSwapchainCreateInfoKHR createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR;
         createInfo.surface = surface;
         createInfo.minImageCount = imageCount;
         createInfo.imageFormat = surfaceFormat.format;
         createInfo.imageColorSpace = surfaceFormat.colorSpace;
         createInfo.imageExtent = extent;
         createInfo.imageArrayLayers = 1;
         createInfo.imageUsage = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT;

         QueueFamilyIndices indices = findQueueFamilies(physicalDevice);
         uint32_t queueFamilyIndices[] = {indices.graphicsFamily.value(), indices.presentFamily.value()};

         if (indices.graphicsFamily != indices.presentFamily) {
            createInfo.imageSharingMode = VK_SHARING_MODE_CONCURRENT;
            createInfo.queueFamilyIndexCount = 2;
            createInfo.pQueueFamilyIndices = queueFamilyIndices;
         } else {
            createInfo.imageSharingMode = VK_SHARING_MODE_EXCLUSIVE;
            createInfo.queueFamilyIndexCount = 0;
            createInfo.pQueueFamilyIndices = nullptr;
         }

         createInfo.preTransform = swapChainSupport.capabilities.currentTransform;
         createInfo.compositeAlpha = VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR;
         createInfo.presentMode = presentMode;
         createInfo.clipped = VK_TRUE;
         createInfo.oldSwapchain = VK_NULL_HANDLE;

         if (vkCreateSwapchainKHR(device, &createInfo, nullptr, &swapChain) != VK_SUCCESS) {
            throw runtime_error("failed to create swap chain!");
         }

         vkGetSwapchainImagesKHR(device, swapChain, &imageCount, nullptr);
         swapChainImages.resize(imageCount);
         vkGetSwapchainImagesKHR(device, swapChain, &imageCount, swapChainImages.data());

         swapChainImageFormat = surfaceFormat.format;
         swapChainExtent = extent;
      }

      SwapChainSupportDetails querySwapChainSupport(VkPhysicalDevice device) {
         SwapChainSupportDetails details;

         vkGetPhysicalDeviceSurfaceCapabilitiesKHR(device, surface, &details.capabilities);

         uint32_t formatCount;
         vkGetPhysicalDeviceSurfaceFormatsKHR(device, surface, &formatCount, nullptr);

         if (formatCount != 0) {
            details.formats.resize(formatCount);
            vkGetPhysicalDeviceSurfaceFormatsKHR(device, surface, &formatCount, details.formats.data());
         }

         uint32_t presentModeCount;
         vkGetPhysicalDeviceSurfacePresentModesKHR(device, surface, &presentModeCount, nullptr);

         if (presentModeCount != 0) {
            details.presentModes.resize(presentModeCount);
            vkGetPhysicalDeviceSurfacePresentModesKHR(device, surface, &presentModeCount, details.presentModes.data());
         }

         return details;
      }

      VkSurfaceFormatKHR chooseSwapSurfaceFormat(const vector<VkSurfaceFormatKHR>& availableFormats) {
         for (const auto& availableFormat : availableFormats) {
            if (availableFormat.format == VK_FORMAT_B8G8R8A8_UNORM && availableFormat.colorSpace == VK_COLOR_SPACE_SRGB_NONLINEAR_KHR) {
               return availableFormat;
            }
         }

         return availableFormats[0];
      }

      VkPresentModeKHR chooseSwapPresentMode(const vector<VkPresentModeKHR>& availablePresentModes) {
         VkPresentModeKHR bestMode = VK_PRESENT_MODE_FIFO_KHR;

         for (const auto& availablePresentMode : availablePresentModes) {
            if (availablePresentMode == VK_PRESENT_MODE_MAILBOX_KHR) {
               return availablePresentMode;
            }
            else if (availablePresentMode == VK_PRESENT_MODE_IMMEDIATE_KHR) {
               bestMode = availablePresentMode;
            }
         }

         return bestMode;
      }

      VkExtent2D chooseSwapExtent(const VkSurfaceCapabilitiesKHR& capabilities) {
         if (capabilities.currentExtent.width != numeric_limits<uint32_t>::max()) {
            return capabilities.currentExtent;
         } else {
            VkExtent2D actualExtent = {
               static_cast<uint32_t>(gui->getWindowWidth()),
               static_cast<uint32_t>(gui->getWindowHeight())
            };

            actualExtent.width = std::max(capabilities.minImageExtent.width, std::min(capabilities.maxImageExtent.width, actualExtent.width));
            actualExtent.height = std::max(capabilities.minImageExtent.height, std::min(capabilities.maxImageExtent.height, actualExtent.height));

            return actualExtent;
         }
      }

      void createImageViews() {
         swapChainImageViews.resize(swapChainImages.size());

         for (size_t i = 0; i < swapChainImages.size(); i++) {
            swapChainImageViews[i] = createImageView(swapChainImages[i], swapChainImageFormat, VK_IMAGE_ASPECT_COLOR_BIT);
         }
      }
/*** END OF REFACTORED CODE ***/

      void createRenderPass() {
         VkAttachmentDescription colorAttachment = {};
         colorAttachment.format = swapChainImageFormat;
         colorAttachment.samples = VK_SAMPLE_COUNT_1_BIT;
         colorAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
         colorAttachment.storeOp = VK_ATTACHMENT_STORE_OP_STORE;
         colorAttachment.stencilLoadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
         colorAttachment.stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
         colorAttachment.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
         colorAttachment.finalLayout = VK_IMAGE_LAYOUT_PRESENT_SRC_KHR;

         VkAttachmentReference colorAttachmentRef = {};
         colorAttachmentRef.attachment = 0;
         colorAttachmentRef.layout = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;

         VkAttachmentDescription depthAttachment = {};
         depthAttachment.format = findDepthFormat();
         depthAttachment.samples = VK_SAMPLE_COUNT_1_BIT;
         depthAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
         depthAttachment.storeOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
         depthAttachment.stencilLoadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
         depthAttachment.stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
         depthAttachment.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
         depthAttachment.finalLayout = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;

         VkAttachmentReference depthAttachmentRef = {};
         depthAttachmentRef.attachment = 1;
         depthAttachmentRef.layout = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;

         VkSubpassDescription subpass = {};
         subpass.pipelineBindPoint = VK_PIPELINE_BIND_POINT_GRAPHICS;
         subpass.colorAttachmentCount = 1;
         subpass.pColorAttachments = &colorAttachmentRef;
         subpass.pDepthStencilAttachment = &depthAttachmentRef;

         VkSubpassDependency dependency = {};
         dependency.srcSubpass = VK_SUBPASS_EXTERNAL;
         dependency.dstSubpass = 0;
         dependency.srcStageMask = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
         dependency.srcAccessMask = 0;
         dependency.dstStageMask = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
         dependency.dstAccessMask = VK_ACCESS_COLOR_ATTACHMENT_READ_BIT | VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;

         array<VkAttachmentDescription, 2> attachments = { colorAttachment, depthAttachment };
         VkRenderPassCreateInfo renderPassInfo = {};
         renderPassInfo.sType = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO;
         renderPassInfo.attachmentCount = static_cast<uint32_t>(attachments.size());
         renderPassInfo.pAttachments = attachments.data();
         renderPassInfo.subpassCount = 1;
         renderPassInfo.pSubpasses = &subpass;
         renderPassInfo.dependencyCount = 1;
         renderPassInfo.pDependencies = &dependency;

         if (vkCreateRenderPass(device, &renderPassInfo, nullptr, &renderPass) != VK_SUCCESS) {
            throw runtime_error("failed to create render pass!");
         }
      }

      void initGraphicsPipelineInfo(GraphicsPipelineInfo& info,
            const void* vertexData, int vertexSize, size_t numVertices,
            const void* indexData, int indexSize, size_t numIndices) {
         // Since there is only one array of vertex data, we use binding = 0
         // I'll probably do that for the foreseeable future
         // I can calculate the stride myself given info about all the varying attributes

         info.bindingDescription.binding = 0;
         info.bindingDescription.stride = vertexSize;
         info.bindingDescription.inputRate = VK_VERTEX_INPUT_RATE_VERTEX;

         info.numVertices = numVertices;
         createVertexBuffer(info, vertexData, vertexSize * numVertices);

         info.numIndices = numIndices;
         createIndexBuffer(info, indexData, indexSize * numIndices);
      }

      void addAttributeDescription(GraphicsPipelineInfo& info, VkFormat format, size_t offset) {
         VkVertexInputAttributeDescription attributeDesc = {};

         attributeDesc.binding = 0;
         attributeDesc.location = info.attributeDescriptions.size();
         attributeDesc.format = format;
         attributeDesc.offset = offset;

         info.attributeDescriptions.push_back(attributeDesc);
      }

      void addDescriptorInfo(GraphicsPipelineInfo& info, VkDescriptorType type, VkShaderStageFlags stageFlags, vector<VkDescriptorBufferInfo>* bufferData, VkDescriptorImageInfo* imageData) {
         info.descriptorInfoList.push_back({ type, stageFlags, bufferData, imageData });
      }

      void createDescriptorSetLayout(GraphicsPipelineInfo& info) {
         vector<VkDescriptorSetLayoutBinding> bindings(info.descriptorInfoList.size());

         for (size_t i = 0; i < bindings.size(); i++) {
            bindings[i].binding = i;
            bindings[i].descriptorCount = 1;
            bindings[i].descriptorType = info.descriptorInfoList[i].type;
            bindings[i].stageFlags = info.descriptorInfoList[i].stageFlags;
            bindings[i].pImmutableSamplers = nullptr;
         }

         VkDescriptorSetLayoutCreateInfo layoutInfo = {};
         layoutInfo.sType = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO;
         layoutInfo.bindingCount = static_cast<uint32_t>(bindings.size());
         layoutInfo.pBindings = bindings.data();

         if (vkCreateDescriptorSetLayout(device, &layoutInfo, nullptr, &info.descriptorSetLayout) != VK_SUCCESS) {
            throw runtime_error("failed to create descriptor set layout!");
         }
      }

      void createGraphicsPipeline(string vertShaderFile, string fragShaderFile, GraphicsPipelineInfo& info) {
         auto vertShaderCode = readFile(vertShaderFile);
         auto fragShaderCode = readFile(fragShaderFile);

         VkShaderModule vertShaderModule = createShaderModule(vertShaderCode);
         VkShaderModule fragShaderModule = createShaderModule(fragShaderCode);

         VkPipelineShaderStageCreateInfo vertShaderStageInfo = {};
         vertShaderStageInfo.sType = VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO;
         vertShaderStageInfo.stage = VK_SHADER_STAGE_VERTEX_BIT;
         vertShaderStageInfo.module = vertShaderModule;
         vertShaderStageInfo.pName = "main";

         VkPipelineShaderStageCreateInfo fragShaderStageInfo = {};
         fragShaderStageInfo.sType = VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO;
         fragShaderStageInfo.stage = VK_SHADER_STAGE_FRAGMENT_BIT;
         fragShaderStageInfo.module = fragShaderModule;
         fragShaderStageInfo.pName = "main";

         VkPipelineShaderStageCreateInfo shaderStages[] = { vertShaderStageInfo, fragShaderStageInfo };

         VkPipelineVertexInputStateCreateInfo vertexInputInfo = {};
         vertexInputInfo.sType = VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO;

         vertexInputInfo.vertexBindingDescriptionCount = 1;
         vertexInputInfo.vertexAttributeDescriptionCount = static_cast<uint32_t>(info.attributeDescriptions.size());
         vertexInputInfo.pVertexBindingDescriptions = &info.bindingDescription;
         vertexInputInfo.pVertexAttributeDescriptions = info.attributeDescriptions.data();

         VkPipelineInputAssemblyStateCreateInfo inputAssembly = {};
         inputAssembly.sType = VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO;
         inputAssembly.topology = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST;
         inputAssembly.primitiveRestartEnable = VK_FALSE;

         VkViewport viewport = {};
         viewport.x = 0.0f;
         viewport.y = 0.0f;
         viewport.width = (float) swapChainExtent.width;
         viewport.height = (float) swapChainExtent.height;
         viewport.minDepth = 0.0f;
         viewport.maxDepth = 1.0f;

         VkRect2D scissor = {};
         scissor.offset = { 0, 0 };
         scissor.extent = swapChainExtent;

         VkPipelineViewportStateCreateInfo viewportState = {};
         viewportState.sType = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO;
         viewportState.viewportCount = 1;
         viewportState.pViewports = &viewport;
         viewportState.scissorCount = 1;
         viewportState.pScissors = &scissor;

         VkPipelineRasterizationStateCreateInfo rasterizer = {};
         rasterizer.sType = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO;
         rasterizer.depthClampEnable = VK_FALSE;
         rasterizer.rasterizerDiscardEnable = VK_FALSE;
         rasterizer.polygonMode = VK_POLYGON_MODE_FILL;
         rasterizer.lineWidth = 1.0f;
         rasterizer.cullMode = VK_CULL_MODE_BACK_BIT;
         rasterizer.frontFace = VK_FRONT_FACE_COUNTER_CLOCKWISE;
         rasterizer.depthBiasEnable = VK_FALSE;

         VkPipelineMultisampleStateCreateInfo multisampling = {};
         multisampling.sType = VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO;
         multisampling.sampleShadingEnable = VK_FALSE;
         multisampling.rasterizationSamples = VK_SAMPLE_COUNT_1_BIT;

         VkPipelineColorBlendAttachmentState colorBlendAttachment = {};
         colorBlendAttachment.colorWriteMask = VK_COLOR_COMPONENT_R_BIT | VK_COLOR_COMPONENT_G_BIT | VK_COLOR_COMPONENT_B_BIT | VK_COLOR_COMPONENT_A_BIT;
         colorBlendAttachment.blendEnable = VK_TRUE;
         colorBlendAttachment.colorBlendOp = VK_BLEND_OP_ADD;
         colorBlendAttachment.srcColorBlendFactor = VK_BLEND_FACTOR_SRC_ALPHA;
         colorBlendAttachment.dstColorBlendFactor = VK_BLEND_FACTOR_ONE_MINUS_SRC_ALPHA;
         colorBlendAttachment.alphaBlendOp = VK_BLEND_OP_ADD;
         colorBlendAttachment.srcAlphaBlendFactor = VK_BLEND_FACTOR_SRC_ALPHA;
         colorBlendAttachment.dstAlphaBlendFactor = VK_BLEND_FACTOR_ONE_MINUS_SRC_ALPHA;

         VkPipelineColorBlendStateCreateInfo colorBlending = {};
         colorBlending.sType = VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO;
         colorBlending.logicOpEnable = VK_FALSE;
         colorBlending.logicOp = VK_LOGIC_OP_COPY;
         colorBlending.attachmentCount = 1;
         colorBlending.pAttachments = &colorBlendAttachment;
         colorBlending.blendConstants[0] = 0.0f;
         colorBlending.blendConstants[1] = 0.0f;
         colorBlending.blendConstants[2] = 0.0f;
         colorBlending.blendConstants[3] = 0.0f;

         VkPipelineDepthStencilStateCreateInfo depthStencil = {};
         depthStencil.sType = VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO;
         depthStencil.depthTestEnable = VK_TRUE;
         depthStencil.depthWriteEnable = VK_TRUE;
         depthStencil.depthCompareOp = VK_COMPARE_OP_LESS;
         depthStencil.depthBoundsTestEnable = VK_FALSE;
         depthStencil.minDepthBounds = 0.0f;
         depthStencil.maxDepthBounds = 1.0f;
         depthStencil.stencilTestEnable = VK_FALSE;
         depthStencil.front = {};
         depthStencil.back = {};

         VkPipelineLayoutCreateInfo pipelineLayoutInfo = {};
         pipelineLayoutInfo.sType = VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO;
         pipelineLayoutInfo.setLayoutCount = 1;
         pipelineLayoutInfo.pSetLayouts = &info.descriptorSetLayout;
         pipelineLayoutInfo.pushConstantRangeCount = 0;

         if (vkCreatePipelineLayout(device, &pipelineLayoutInfo, nullptr, &info.pipelineLayout) != VK_SUCCESS) {
            throw runtime_error("failed to create pipeline layout!");
         }

         VkGraphicsPipelineCreateInfo pipelineInfo = {};
         pipelineInfo.sType = VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO;
         pipelineInfo.stageCount = 2;
         pipelineInfo.pStages = shaderStages;
         pipelineInfo.pVertexInputState = &vertexInputInfo;
         pipelineInfo.pInputAssemblyState = &inputAssembly;
         pipelineInfo.pViewportState = &viewportState;
         pipelineInfo.pRasterizationState = &rasterizer;
         pipelineInfo.pMultisampleState = &multisampling;
         pipelineInfo.pDepthStencilState = &depthStencil;
         pipelineInfo.pColorBlendState = &colorBlending;
         pipelineInfo.pDynamicState = nullptr;
         pipelineInfo.layout = info.pipelineLayout;
         pipelineInfo.renderPass = renderPass;
         pipelineInfo.subpass = 0;
         pipelineInfo.basePipelineHandle = VK_NULL_HANDLE;
         pipelineInfo.basePipelineIndex = -1;

         if (vkCreateGraphicsPipelines(device, VK_NULL_HANDLE, 1, &pipelineInfo, nullptr, &info.pipeline) != VK_SUCCESS) {
            throw runtime_error("failed to create graphics pipeline!");
         }

         vkDestroyShaderModule(device, vertShaderModule, nullptr);
         vkDestroyShaderModule(device, fragShaderModule, nullptr);
      }

      VkShaderModule createShaderModule(const vector<char>& code) {
         VkShaderModuleCreateInfo createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO;
         createInfo.codeSize = code.size();
         createInfo.pCode = reinterpret_cast<const uint32_t*>(code.data());

         VkShaderModule shaderModule;
         if (vkCreateShaderModule(device, &createInfo, nullptr, &shaderModule) != VK_SUCCESS) {
            throw runtime_error("failed to create shader module!");
         }

         return shaderModule;
      }

      void createFramebuffers() {
         swapChainFramebuffers.resize(swapChainImageViews.size());

         for (size_t i = 0; i < swapChainImageViews.size(); i++) {
            array <VkImageView, 2> attachments = {
               swapChainImageViews[i],
               depthImageView
            };

            VkFramebufferCreateInfo framebufferInfo = {};
            framebufferInfo.sType = VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO;
            framebufferInfo.renderPass = renderPass;
            framebufferInfo.attachmentCount = static_cast<uint32_t>(attachments.size());
            framebufferInfo.pAttachments = attachments.data();
            framebufferInfo.width = swapChainExtent.width;
            framebufferInfo.height = swapChainExtent.height;
            framebufferInfo.layers = 1;

            if (vkCreateFramebuffer(device, &framebufferInfo, nullptr, &swapChainFramebuffers[i]) != VK_SUCCESS) {
               throw runtime_error("failed to create framebuffer!");
            }
         }
      }

      void createCommandPool() {
         QueueFamilyIndices queueFamilyIndices = findQueueFamilies(physicalDevice);

         VkCommandPoolCreateInfo poolInfo = {};
         poolInfo.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
         poolInfo.queueFamilyIndex = queueFamilyIndices.graphicsFamily.value();
         poolInfo.flags = 0;

         if (vkCreateCommandPool(device, &poolInfo, nullptr, &commandPool) != VK_SUCCESS) {
            throw runtime_error("failed to create graphics command pool!");
         }
      }

/*** START OF REFACTORED CODE ***/
      QueueFamilyIndices findQueueFamilies(VkPhysicalDevice device) {
         QueueFamilyIndices indices;

         uint32_t queueFamilyCount = 0;
         vkGetPhysicalDeviceQueueFamilyProperties(device, &queueFamilyCount, nullptr);

         vector<VkQueueFamilyProperties> queueFamilies(queueFamilyCount);
         vkGetPhysicalDeviceQueueFamilyProperties(device, &queueFamilyCount, queueFamilies.data());

         int i = 0;
         for (const auto& queueFamily : queueFamilies) {
            if (queueFamily.queueCount > 0 && queueFamily.queueFlags & VK_QUEUE_GRAPHICS_BIT) {
               indices.graphicsFamily = i;
            }

            VkBool32 presentSupport = false;
            vkGetPhysicalDeviceSurfaceSupportKHR(device, i, surface, &presentSupport);

            if (queueFamily.queueCount > 0 && presentSupport) {
               indices.presentFamily = i;
            }

            if (indices.isComplete()) {
               break;
            }

            i++;
         }

         return indices;
      }
/*** END OF REFACTORED CODE ***/

      void createDepthResources() {
         VkFormat depthFormat = findDepthFormat();

         createImage(swapChainExtent.width, swapChainExtent.height, depthFormat, VK_IMAGE_TILING_OPTIMAL,
            VK_IMAGE_USAGE_DEPTH_STENCIL_ATTACHMENT_BIT, VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT, depthImage, depthImageMemory);
         depthImageView = createImageView(depthImage, depthFormat, VK_IMAGE_ASPECT_DEPTH_BIT);

         transitionImageLayout(depthImage, depthFormat, VK_IMAGE_LAYOUT_UNDEFINED, VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL);
      }

      VkFormat findDepthFormat() {
         return findSupportedFormat(
            { VK_FORMAT_D32_SFLOAT, VK_FORMAT_D32_SFLOAT_S8_UINT, VK_FORMAT_D24_UNORM_S8_UINT },
            VK_IMAGE_TILING_OPTIMAL,
            VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT
         );
      }

      VkFormat findSupportedFormat(const vector<VkFormat>& candidates, VkImageTiling tiling, 
            VkFormatFeatureFlags features) {
         for (VkFormat format : candidates) {
            VkFormatProperties props;
            vkGetPhysicalDeviceFormatProperties(physicalDevice, format, &props);

            if (tiling == VK_IMAGE_TILING_LINEAR &&
                  (props.linearTilingFeatures & features) == features) {
               return format;
            } else if (tiling == VK_IMAGE_TILING_OPTIMAL &&
                  (props.optimalTilingFeatures & features) == features) {
               return format;
            }
         }

         throw runtime_error("failed to find supported format!");
      }

      bool hasStencilComponent(VkFormat format) {
         return format == VK_FORMAT_D32_SFLOAT_S8_UINT || format == VK_FORMAT_D24_UNORM_S8_UINT;
      }

      void createImageResources(string filename, VkImage& image, VkDeviceMemory& imageMemory, VkImageView& view) {
         int texWidth, texHeight, texChannels;

         stbi_uc* pixels = stbi_load(filename.c_str(), &texWidth, &texHeight, &texChannels, STBI_rgb_alpha);
         VkDeviceSize imageSize = texWidth * texHeight * 4;

         if (!pixels) {
            throw runtime_error("failed to load texture image!");
         }

         VkBuffer stagingBuffer;
         VkDeviceMemory stagingBufferMemory;

         createBuffer(imageSize, VK_BUFFER_USAGE_TRANSFER_SRC_BIT,
            VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT,
            stagingBuffer, stagingBufferMemory);

         void* data;

         vkMapMemory(device, stagingBufferMemory, 0, imageSize, 0, &data);
         memcpy(data, pixels, static_cast<size_t>(imageSize));
         vkUnmapMemory(device, stagingBufferMemory);

         stbi_image_free(pixels);

         createImage(texWidth, texHeight, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_TILING_OPTIMAL,
            VK_IMAGE_USAGE_TRANSFER_DST_BIT | VK_IMAGE_USAGE_SAMPLED_BIT,
            VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT, image, imageMemory);

         transitionImageLayout(image, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_LAYOUT_UNDEFINED, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL);
         copyBufferToImage(stagingBuffer, image, static_cast<uint32_t>(texWidth), static_cast<uint32_t>(texHeight));
         transitionImageLayout(image, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL);

         vkDestroyBuffer(device, stagingBuffer, nullptr);
         vkFreeMemory(device, stagingBufferMemory, nullptr);

         view = createImageView(image, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_ASPECT_COLOR_BIT);
      }

      void createImageResourcesFromSDLTexture(SDL_Texture* texture, VkImage& image, VkDeviceMemory& imageMemory, VkImageView& view) {
         int a, w, h;

         // I only need this here for the width and height, which are constants, so just use those instead
         SDL_QueryTexture(texture, nullptr, &a, &w, &h);

         createImage(w, h, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_TILING_OPTIMAL,
            VK_IMAGE_USAGE_TRANSFER_DST_BIT | VK_IMAGE_USAGE_SAMPLED_BIT,
            VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT, image, imageMemory);

         view = createImageView(image, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_ASPECT_COLOR_BIT);
      }

      void populateImageFromSDLTexture(SDL_Texture* texture, VkImage& image) {
         int a, w, h;

         SDL_QueryTexture(texture, nullptr, &a, &w, &h);

         VkDeviceSize imageSize = w * h * 4;
         unsigned char* pixels = new unsigned char[imageSize];

         SDL_RenderReadPixels(gRenderer, nullptr, SDL_PIXELFORMAT_ABGR8888, pixels, w * 4);

         VkBuffer stagingBuffer;
         VkDeviceMemory stagingBufferMemory;

         createBuffer(imageSize,
            VK_BUFFER_USAGE_TRANSFER_DST_BIT | VK_BUFFER_USAGE_TRANSFER_SRC_BIT,
            VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT,
            stagingBuffer, stagingBufferMemory);

         void* data;

         vkMapMemory(device, stagingBufferMemory, 0, VK_WHOLE_SIZE, 0, &data);
         memcpy(data, pixels, static_cast<size_t>(imageSize));

         VkMappedMemoryRange mappedMemoryRange = {};
         mappedMemoryRange.sType = VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE;
         mappedMemoryRange.memory = stagingBufferMemory;
         mappedMemoryRange.offset = 0;
         mappedMemoryRange.size = VK_WHOLE_SIZE;

         // TODO: Should probably check that the function succeeded
         vkFlushMappedMemoryRanges(device, 1, &mappedMemoryRange);
         vkUnmapMemory(device, stagingBufferMemory);

         delete[] pixels;

         transitionImageLayout(image, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_LAYOUT_UNDEFINED, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL);
         copyBufferToImage(stagingBuffer, image, static_cast<uint32_t>(w), static_cast<uint32_t>(h));
         transitionImageLayout(image, VK_FORMAT_R8G8B8A8_UNORM, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL);

         vkDestroyBuffer(device, stagingBuffer, nullptr);
         vkFreeMemory(device, stagingBufferMemory, nullptr);
      }

      void createImage(uint32_t width, uint32_t height, VkFormat format, VkImageTiling tiling, VkImageUsageFlags usage,
            VkMemoryPropertyFlags properties, VkImage& image, VkDeviceMemory& imageMemory) {
         VkImageCreateInfo imageInfo = {};
         imageInfo.sType = VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO;
         imageInfo.imageType = VK_IMAGE_TYPE_2D;
         imageInfo.extent.width = width;
         imageInfo.extent.height = height;
         imageInfo.extent.depth = 1;
         imageInfo.mipLevels = 1;
         imageInfo.arrayLayers = 1;
         imageInfo.format = format;
         imageInfo.tiling = tiling;
         imageInfo.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
         imageInfo.usage = usage;
         imageInfo.samples = VK_SAMPLE_COUNT_1_BIT;
         imageInfo.sharingMode = VK_SHARING_MODE_EXCLUSIVE;

         if (vkCreateImage(device, &imageInfo, nullptr, &image) != VK_SUCCESS) {
            throw runtime_error("failed to create image!");
         }

         VkMemoryRequirements memRequirements;
         vkGetImageMemoryRequirements(device, image, &memRequirements);

         VkMemoryAllocateInfo allocInfo = {};
         allocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
         allocInfo.allocationSize = memRequirements.size;
         allocInfo.memoryTypeIndex = findMemoryType(memRequirements.memoryTypeBits, properties);

         if (vkAllocateMemory(device, &allocInfo, nullptr, &imageMemory) != VK_SUCCESS) {
            throw runtime_error("failed to allocate image memory!");
         }

         vkBindImageMemory(device, image, imageMemory, 0);
      }

      void transitionImageLayout(VkImage image, VkFormat format, VkImageLayout oldLayout, VkImageLayout newLayout) {
         VkCommandBuffer commandBuffer = beginSingleTimeCommands();

         VkImageMemoryBarrier barrier = {};
         barrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
         barrier.oldLayout = oldLayout;
         barrier.newLayout = newLayout;
         barrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
         barrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
         barrier.image = image;

         if (newLayout == VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL) {
            barrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_DEPTH_BIT;

            if (hasStencilComponent(format)) {
               barrier.subresourceRange.aspectMask |= VK_IMAGE_ASPECT_STENCIL_BIT;
            }
         } else {
            barrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
         }

         barrier.subresourceRange.baseMipLevel = 0;
         barrier.subresourceRange.levelCount = 1;
         barrier.subresourceRange.baseArrayLayer = 0;
         barrier.subresourceRange.layerCount = 1;

         VkPipelineStageFlags sourceStage;
         VkPipelineStageFlags destinationStage;

         if (oldLayout == VK_IMAGE_LAYOUT_UNDEFINED && newLayout == VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL) {
            barrier.srcAccessMask = 0;
            barrier.dstAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;

            sourceStage = VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT;
            destinationStage = VK_PIPELINE_STAGE_TRANSFER_BIT;
         } else if (oldLayout == VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL && newLayout == VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL) {
            barrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
            barrier.dstAccessMask = VK_ACCESS_SHADER_READ_BIT;

            sourceStage = VK_PIPELINE_STAGE_TRANSFER_BIT;
            destinationStage = VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT;
         } else if (oldLayout == VK_IMAGE_LAYOUT_UNDEFINED && newLayout == VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL) {
            barrier.srcAccessMask = 0;
            barrier.dstAccessMask = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT | VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT;

            sourceStage = VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT;
            destinationStage = VK_PIPELINE_STAGE_EARLY_FRAGMENT_TESTS_BIT;
         } else {
            throw invalid_argument("unsupported layout transition!");
         }

         vkCmdPipelineBarrier(
            commandBuffer,
            sourceStage, destinationStage,
            0,
            0, nullptr,
            0, nullptr,
            1, &barrier
         );

         endSingleTimeCommands(commandBuffer);
      }

      void copyBufferToImage(VkBuffer buffer, VkImage image, uint32_t width, uint32_t height) {
         VkCommandBuffer commandBuffer = beginSingleTimeCommands();

         VkBufferImageCopy region = {};
         region.bufferOffset = 0;
         region.bufferRowLength = 0;
         region.bufferImageHeight = 0;
         region.imageSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
         region.imageSubresource.mipLevel = 0;
         region.imageSubresource.baseArrayLayer = 0;
         region.imageSubresource.layerCount = 1;
         region.imageOffset = { 0, 0, 0 };
         region.imageExtent = { width, height, 1 };

         vkCmdCopyBufferToImage(
            commandBuffer,
            buffer,
            image,
            VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL,
            1,
            &region
         );

         endSingleTimeCommands(commandBuffer);
      }

/*** START OF REFACTORED CODE ***/
      VkImageView createImageView(VkImage image, VkFormat format, VkImageAspectFlags aspectFlags) {
         VkImageViewCreateInfo viewInfo = {};
         viewInfo.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
         viewInfo.image = image;
         viewInfo.viewType = VK_IMAGE_VIEW_TYPE_2D;
         viewInfo.format = format;

         viewInfo.components.r = VK_COMPONENT_SWIZZLE_IDENTITY;
         viewInfo.components.g = VK_COMPONENT_SWIZZLE_IDENTITY;
         viewInfo.components.b = VK_COMPONENT_SWIZZLE_IDENTITY;
         viewInfo.components.a = VK_COMPONENT_SWIZZLE_IDENTITY;

         viewInfo.subresourceRange.aspectMask = aspectFlags;
         viewInfo.subresourceRange.baseMipLevel = 0;
         viewInfo.subresourceRange.levelCount = 1;
         viewInfo.subresourceRange.baseArrayLayer = 0;
         viewInfo.subresourceRange.layerCount = 1;

         VkImageView imageView;
         if (vkCreateImageView(device, &viewInfo, nullptr, &imageView) != VK_SUCCESS) {
            throw runtime_error("failed to create texture image view!");
         }

         return imageView;
      }
/*** END OF REFACTORED CODE ***/

      void createTextureSampler() {
         VkSamplerCreateInfo samplerInfo = {};
         samplerInfo.sType = VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO;
         samplerInfo.magFilter = VK_FILTER_LINEAR;
         samplerInfo.minFilter = VK_FILTER_LINEAR;

         samplerInfo.addressModeU = VK_SAMPLER_ADDRESS_MODE_REPEAT;
         samplerInfo.addressModeV = VK_SAMPLER_ADDRESS_MODE_REPEAT;
         samplerInfo.addressModeW = VK_SAMPLER_ADDRESS_MODE_REPEAT;

         samplerInfo.anisotropyEnable = VK_TRUE;
         samplerInfo.maxAnisotropy = 16;
         samplerInfo.borderColor = VK_BORDER_COLOR_INT_OPAQUE_BLACK;
         samplerInfo.unnormalizedCoordinates = VK_FALSE;
         samplerInfo.compareEnable = VK_FALSE;
         samplerInfo.compareOp = VK_COMPARE_OP_ALWAYS;
         samplerInfo.mipmapMode = VK_SAMPLER_MIPMAP_MODE_LINEAR;
         samplerInfo.mipLodBias = 0.0f;
         samplerInfo.minLod = 0.0f;
         samplerInfo.maxLod = 0.0f;

         if (vkCreateSampler(device, &samplerInfo, nullptr, &textureSampler) != VK_SUCCESS) {
            throw runtime_error("failed to create texture sampler!");
         }
      }

      void createVertexBuffer(GraphicsPipelineInfo& info, const void* vertexData, VkDeviceSize bufferSize) {
         VkBuffer stagingBuffer;
         VkDeviceMemory stagingBufferMemory;
         createBuffer(bufferSize, VK_BUFFER_USAGE_TRANSFER_SRC_BIT,
            VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT,
            stagingBuffer, stagingBufferMemory);

         void* data;
         vkMapMemory(device, stagingBufferMemory, 0, bufferSize, 0, &data);
         memcpy(data, vertexData, (size_t) bufferSize);
         vkUnmapMemory(device, stagingBufferMemory);

         createBuffer(bufferSize, VK_BUFFER_USAGE_TRANSFER_DST_BIT | VK_BUFFER_USAGE_VERTEX_BUFFER_BIT,
            VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT, info.vertexBuffer, info.vertexBufferMemory);

         copyBuffer(stagingBuffer, info.vertexBuffer, bufferSize);

         vkDestroyBuffer(device, stagingBuffer, nullptr);
         vkFreeMemory(device, stagingBufferMemory, nullptr);
      }

      void createIndexBuffer(GraphicsPipelineInfo& info, const void* indexData, VkDeviceSize bufferSize) {
         VkBuffer stagingBuffer;
         VkDeviceMemory stagingBufferMemory;
         createBuffer(bufferSize, VK_BUFFER_USAGE_TRANSFER_SRC_BIT,
            VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT,
            stagingBuffer, stagingBufferMemory);

         void* data;
         vkMapMemory(device, stagingBufferMemory, 0, bufferSize, 0, &data);
         memcpy(data, indexData, (size_t) bufferSize);
         vkUnmapMemory(device, stagingBufferMemory);

         createBuffer(bufferSize, VK_BUFFER_USAGE_TRANSFER_DST_BIT | VK_BUFFER_USAGE_INDEX_BUFFER_BIT,
            VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT, info.indexBuffer, info.indexBufferMemory);

         copyBuffer(stagingBuffer, info.indexBuffer, bufferSize);

         vkDestroyBuffer(device, stagingBuffer, nullptr);
         vkFreeMemory(device, stagingBufferMemory, nullptr);
      }

      void createUniformBuffers() {
         VkDeviceSize bufferSize = sizeof(UniformBufferObject);

         uniformBuffers.resize(swapChainImages.size());
         uniformBuffersMemory.resize(swapChainImages.size());
         uniformBufferInfoList.resize(swapChainImages.size());

         for (size_t i = 0; i < swapChainImages.size(); i++) {
            createBuffer(bufferSize, VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT,
               VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT,
               uniformBuffers[i], uniformBuffersMemory[i]);

            uniformBufferInfoList[i].buffer = uniformBuffers[i];
            uniformBufferInfoList[i].offset = 0;
            uniformBufferInfoList[i].range = sizeof(UniformBufferObject);
         }
      }

      void createBuffer(VkDeviceSize size, VkBufferUsageFlags usage, VkMemoryPropertyFlags properties, VkBuffer& buffer, VkDeviceMemory& bufferMemory) {
         VkBufferCreateInfo bufferInfo = {};
         bufferInfo.sType = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO;
         bufferInfo.size = size;
         bufferInfo.usage = usage;
         bufferInfo.sharingMode = VK_SHARING_MODE_EXCLUSIVE;

         if (vkCreateBuffer(device, &bufferInfo, nullptr, &buffer) != VK_SUCCESS) {
            throw runtime_error("failed to create buffer!");
         }

         VkMemoryRequirements memRequirements;
         vkGetBufferMemoryRequirements(device, buffer, &memRequirements);

         VkMemoryAllocateInfo allocInfo = {};
         allocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
         allocInfo.allocationSize = memRequirements.size;
         allocInfo.memoryTypeIndex = findMemoryType(memRequirements.memoryTypeBits, properties);

         if (vkAllocateMemory(device, &allocInfo, nullptr, &bufferMemory) != VK_SUCCESS) {
            throw runtime_error("failed to allocate buffer memory!");
         }

         vkBindBufferMemory(device, buffer, bufferMemory, 0);
      }

      void copyBuffer(VkBuffer srcBuffer, VkBuffer dstBuffer, VkDeviceSize size) {
         VkCommandBuffer commandBuffer = beginSingleTimeCommands();

         VkBufferCopy copyRegion = {};
         copyRegion.size = size;
         vkCmdCopyBuffer(commandBuffer, srcBuffer, dstBuffer, 1, &copyRegion);

         endSingleTimeCommands(commandBuffer);
      }

      VkCommandBuffer beginSingleTimeCommands() {
         VkCommandBufferAllocateInfo allocInfo = {};
         allocInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
         allocInfo.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
         allocInfo.commandPool = commandPool;
         allocInfo.commandBufferCount = 1;

         VkCommandBuffer commandBuffer;
         vkAllocateCommandBuffers(device, &allocInfo, &commandBuffer);

         VkCommandBufferBeginInfo beginInfo = {};
         beginInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
         beginInfo.flags = VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT;

         vkBeginCommandBuffer(commandBuffer, &beginInfo);

         return commandBuffer;
      }

      void endSingleTimeCommands(VkCommandBuffer commandBuffer) {
         vkEndCommandBuffer(commandBuffer);

         VkSubmitInfo submitInfo = {};
         submitInfo.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
         submitInfo.commandBufferCount = 1;
         submitInfo.pCommandBuffers = &commandBuffer;

         vkQueueSubmit(graphicsQueue, 1, &submitInfo, VK_NULL_HANDLE);
         vkQueueWaitIdle(graphicsQueue);

         vkFreeCommandBuffers(device, commandPool, 1, &commandBuffer);
      }

      uint32_t findMemoryType(uint32_t typeFilter, VkMemoryPropertyFlags properties) {
         VkPhysicalDeviceMemoryProperties memProperties;
         vkGetPhysicalDeviceMemoryProperties(physicalDevice, &memProperties);

         for (uint32_t i = 0; i < memProperties.memoryTypeCount; i++) {
            if ((typeFilter & (1 << i)) && (memProperties.memoryTypes[i].propertyFlags & properties) == properties) {
               return i;
            }
         }

         throw runtime_error("failed to find suitable memory type!");
      }

      void createDescriptorPool(GraphicsPipelineInfo& info) {
         vector<VkDescriptorPoolSize> poolSizes(info.descriptorInfoList.size());

         for (size_t i = 0; i < poolSizes.size(); i++) {
            poolSizes[i].type = info.descriptorInfoList[i].type;
            poolSizes[i].descriptorCount = static_cast<uint32_t>(swapChainImages.size());
         }

         VkDescriptorPoolCreateInfo poolInfo = {};
         poolInfo.sType = VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO;
         poolInfo.poolSizeCount = static_cast<uint32_t>(poolSizes.size());
         poolInfo.pPoolSizes = poolSizes.data();
         poolInfo.maxSets = static_cast<uint32_t>(swapChainImages.size());

         if (vkCreateDescriptorPool(device, &poolInfo, nullptr, &info.descriptorPool) != VK_SUCCESS) {
            throw runtime_error("failed to create descriptor pool!");
         }
      }

      void createDescriptorSets(GraphicsPipelineInfo& info) {
         vector<VkDescriptorSetLayout> layouts(swapChainImages.size(), info.descriptorSetLayout);

         VkDescriptorSetAllocateInfo allocInfo = {};
         allocInfo.sType = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO;
         allocInfo.descriptorPool = info.descriptorPool;
         allocInfo.descriptorSetCount = static_cast<uint32_t>(swapChainImages.size());
         allocInfo.pSetLayouts = layouts.data();

         info.descriptorSets.resize(swapChainImages.size());
         if (vkAllocateDescriptorSets(device, &allocInfo, info.descriptorSets.data()) != VK_SUCCESS) {
            throw runtime_error("failed to allocate descriptor sets!");
         }

         for (size_t i = 0; i < swapChainImages.size(); i++) {
            vector<VkWriteDescriptorSet> descriptorWrites(info.descriptorInfoList.size());

            for (size_t j = 0; j < descriptorWrites.size(); j++) {
               descriptorWrites[j].sType = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET;
               descriptorWrites[j].dstSet = info.descriptorSets[i];
               descriptorWrites[j].dstBinding = j;
               descriptorWrites[j].dstArrayElement = 0;
               descriptorWrites[j].descriptorType = info.descriptorInfoList[j].type;
               descriptorWrites[j].descriptorCount = 1;
               descriptorWrites[j].pBufferInfo = nullptr;
               descriptorWrites[j].pImageInfo = nullptr;
               descriptorWrites[j].pTexelBufferView = nullptr;

               switch (descriptorWrites[j].descriptorType) {
                  case VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER:
                     descriptorWrites[j].pBufferInfo = &(*info.descriptorInfoList[j].bufferDataList)[i];
                     break;
                  case VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER:
                     descriptorWrites[j].pImageInfo = info.descriptorInfoList[j].imageData;
                     break;
                  default:
                     cout << "Unknown descriptor type: " << descriptorWrites[j].descriptorType << endl;
               }
            }

            vkUpdateDescriptorSets(device, static_cast<uint32_t>(descriptorWrites.size()), descriptorWrites.data(), 0, nullptr);
         }
      }

      void createCommandBuffers() {
         commandBuffers.resize(swapChainFramebuffers.size());

         VkCommandBufferAllocateInfo allocInfo = {};
         allocInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
         allocInfo.commandPool = commandPool;
         allocInfo.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
         allocInfo.commandBufferCount = (uint32_t) commandBuffers.size();

         if (vkAllocateCommandBuffers(device, &allocInfo, commandBuffers.data()) != VK_SUCCESS) {
            throw runtime_error("failed to allocate command buffers!");
         }

         for (size_t i = 0; i < commandBuffers.size(); i++) {
            VkCommandBufferBeginInfo beginInfo = {};
            beginInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
            beginInfo.flags = VK_COMMAND_BUFFER_USAGE_SIMULTANEOUS_USE_BIT;
            beginInfo.pInheritanceInfo = nullptr;

            if (vkBeginCommandBuffer(commandBuffers[i], &beginInfo) != VK_SUCCESS) {
               throw runtime_error("failed to begin recording command buffer!");
            }

            VkRenderPassBeginInfo renderPassInfo = {};
            renderPassInfo.sType = VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO;
            renderPassInfo.renderPass = renderPass;
            renderPassInfo.framebuffer = swapChainFramebuffers[i];
            renderPassInfo.renderArea.offset = { 0, 0 };
            renderPassInfo.renderArea.extent = swapChainExtent;

            array<VkClearValue, 2> clearValues = {};
            clearValues[0].color = {{ 0.0f, 0.0f, 0.0f, 1.0f }};
            clearValues[1].depthStencil = { 1.0f, 0 };

            renderPassInfo.clearValueCount = static_cast<uint32_t>(clearValues.size());
            renderPassInfo.pClearValues = clearValues.data();

            vkCmdBeginRenderPass(commandBuffers[i], &renderPassInfo, VK_SUBPASS_CONTENTS_INLINE);

            createGraphicsPipelineCommands(scenePipeline, i);
            createGraphicsPipelineCommands(overlayPipeline, i);

            vkCmdEndRenderPass(commandBuffers[i]);

            if (vkEndCommandBuffer(commandBuffers[i]) != VK_SUCCESS) {
               throw runtime_error("failed to record command buffer!");
            }
         }
      }

      void createGraphicsPipelineCommands(GraphicsPipelineInfo& info, uint32_t currentImage) {
         vkCmdBindPipeline(commandBuffers[currentImage], VK_PIPELINE_BIND_POINT_GRAPHICS, info.pipeline);
         vkCmdBindDescriptorSets(commandBuffers[currentImage], VK_PIPELINE_BIND_POINT_GRAPHICS, info.pipelineLayout, 0, 1,
            &info.descriptorSets[currentImage], 0, nullptr);

         VkBuffer vertexBuffers[] = { info.vertexBuffer };
         VkDeviceSize offsets[] = { 0 };
         vkCmdBindVertexBuffers(commandBuffers[currentImage], 0, 1, vertexBuffers, offsets);

         vkCmdBindIndexBuffer(commandBuffers[currentImage], info.indexBuffer, 0, VK_INDEX_TYPE_UINT16);

         vkCmdDrawIndexed(commandBuffers[currentImage], static_cast<uint32_t>(info.numIndices), 1, 0, 0, 0);
      }

      void createSyncObjects() {
         imageAvailableSemaphores.resize(MAX_FRAMES_IN_FLIGHT);
         renderFinishedSemaphores.resize(MAX_FRAMES_IN_FLIGHT);
         inFlightFences.resize(MAX_FRAMES_IN_FLIGHT);

         VkSemaphoreCreateInfo semaphoreInfo = {};
         semaphoreInfo.sType = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO;

         VkFenceCreateInfo fenceInfo = {};
         fenceInfo.sType = VK_STRUCTURE_TYPE_FENCE_CREATE_INFO;
         fenceInfo.flags = VK_FENCE_CREATE_SIGNALED_BIT;

         for (size_t i = 0; i < MAX_FRAMES_IN_FLIGHT; i++) {
            if (vkCreateSemaphore(device, &semaphoreInfo, nullptr, &imageAvailableSemaphores[i]) != VK_SUCCESS ||
                vkCreateSemaphore(device, &semaphoreInfo, nullptr, &renderFinishedSemaphores[i]) != VK_SUCCESS ||
                vkCreateFence(device, &fenceInfo, nullptr, &inFlightFences[i]) != VK_SUCCESS) {
               throw runtime_error("failed to create synchronization objects for a frame!");
            }
         }
      }

/*** START OF REFACTORED CODE ***/
      void mainLoop() {
         // TODO: Create some generic event-handling functions in game-gui-*
         SDL_Event e;
         bool quit = false;

         while (!quit) {
            while (SDL_PollEvent(&e)) {
               if (e.type == SDL_QUIT) {
                  quit = true;
               }
               if (e.type == SDL_KEYDOWN) {
                  if (e.key.keysym.scancode == SDL_SCANCODE_ESCAPE) {
                     quit = true;
                  }
               }
               if (e.type == SDL_MOUSEBUTTONDOWN) {
                  quit = true;
               }
/*** END OF REFACTORED CODE ***/
               if (e.type == SDL_WINDOWEVENT) {
                  if (e.window.event == SDL_WINDOWEVENT_SIZE_CHANGED ||
                      e.window.event == SDL_WINDOWEVENT_MINIMIZED) {
                     framebufferResized = true;
                  }
               }
/*** START OF REFACTORED CODE ***/
            }

            drawUI();

            drawFrame();
         }

         vkDeviceWaitIdle(device);
      }

      void drawFrame() {
/*** END OF REFACTORED CODE ***/
         vkWaitForFences(device, 1, &inFlightFences[currentFrame], VK_TRUE, numeric_limits<uint64_t>::max());

         uint32_t imageIndex;

         VkResult result = vkAcquireNextImageKHR(device, swapChain, numeric_limits<uint64_t>::max(),
            imageAvailableSemaphores[currentFrame], VK_NULL_HANDLE, &imageIndex);

         if (result == VK_ERROR_OUT_OF_DATE_KHR) {
            recreateSwapChain();
            return;
         } else if (result != VK_SUCCESS && result != VK_SUBOPTIMAL_KHR) {
            throw runtime_error("failed to acquire swap chain image!");
         }

         updateUniformBuffer(imageIndex);

         VkSubmitInfo submitInfo = {};
         submitInfo.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;

         VkSemaphore waitSemaphores[] = { imageAvailableSemaphores[currentFrame] };
         VkPipelineStageFlags waitStages[] = { VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT };

         submitInfo.waitSemaphoreCount = 1;
         submitInfo.pWaitSemaphores = waitSemaphores;
         submitInfo.pWaitDstStageMask = waitStages;
         submitInfo.commandBufferCount = 1;
         submitInfo.pCommandBuffers = &commandBuffers[imageIndex];

         VkSemaphore signalSemaphores[] = { renderFinishedSemaphores[currentFrame] };

         submitInfo.signalSemaphoreCount = 1;
         submitInfo.pSignalSemaphores = signalSemaphores;

         vkResetFences(device, 1, &inFlightFences[currentFrame]);

         if (vkQueueSubmit(graphicsQueue, 1, &submitInfo, inFlightFences[currentFrame]) != VK_SUCCESS) {
            throw runtime_error("failed to submit draw command buffer!");
         }

         VkPresentInfoKHR presentInfo = {};
         presentInfo.sType = VK_STRUCTURE_TYPE_PRESENT_INFO_KHR;
         presentInfo.waitSemaphoreCount = 1;
         presentInfo.pWaitSemaphores = signalSemaphores;

         VkSwapchainKHR swapChains[] = { swapChain };
         presentInfo.swapchainCount = 1;
         presentInfo.pSwapchains = swapChains;
         presentInfo.pImageIndices = &imageIndex;
         presentInfo.pResults = nullptr;

         result = vkQueuePresentKHR(presentQueue, &presentInfo);

         if (result == VK_ERROR_OUT_OF_DATE_KHR || result == VK_SUBOPTIMAL_KHR || framebufferResized) {
            framebufferResized = false;
            recreateSwapChain();
         } else if (result != VK_SUCCESS) {
            throw runtime_error("failed to present swap chain image!");
         }

         currentFrame = (currentFrame + 1) % MAX_FRAMES_IN_FLIGHT;
         currentFrame = (currentFrame + 1) % MAX_FRAMES_IN_FLIGHT;
/*** START OF REFACTORED CODE ***/
      }

      void drawUI() {
/*** END OF REFACTORED CODE ***/
         // TODO: Since I currently don't use any other render targets,
         // I may as well set this once before the render loop
         SDL_SetRenderTarget(gRenderer, uiOverlay);

         SDL_SetRenderDrawColor(gRenderer, 0x00, 0x00, 0x00, 0x00);
         SDL_RenderClear(gRenderer);

         SDL_Rect rect;

         rect = {280, 220, 100, 100};
         SDL_SetRenderDrawColor(gRenderer, 0x00, 0xFF, 0x00, 0xFF);
         SDL_RenderFillRect(gRenderer, &rect);
         SDL_SetRenderDrawColor(gRenderer, 0x00, 0x9F, 0x9F, 0xFF);

         rect = {10, 10, 0, 0};
         SDL_QueryTexture(uiText, nullptr, nullptr, &(rect.w), &(rect.h));
         SDL_RenderCopy(gRenderer, uiText, nullptr, &rect);

         rect = {10, 80, 0, 0};
         SDL_QueryTexture(uiImage, nullptr, nullptr, &(rect.w), &(rect.h));
         SDL_RenderCopy(gRenderer, uiImage, nullptr, &rect);

         SDL_SetRenderDrawColor(gRenderer, 0x00, 0x00, 0xFF, 0xFF);
         SDL_RenderDrawLine(gRenderer, 50, 5, 150, 500);

         populateImageFromSDLTexture(uiOverlay, sdlOverlayImage);
/*** START OF REFACTORED CODE ***/
      }
/*** END OF REFACTORED CODE ***/

      void updateUniformBuffer(uint32_t currentImage) {
         static auto startTime = chrono::high_resolution_clock::now();

         auto currentTime = chrono::high_resolution_clock::now();
         float time = chrono::duration<float, chrono::seconds::period>(currentTime - startTime).count();

         UniformBufferObject ubo = {};
         ubo.model = rotate(glm::mat4(1.0f), time * glm::radians(90.0f), glm::vec3(0.0f, 0.0f, 1.0f));
         ubo.view = lookAt(glm::vec3(0.0f, 2.0f, 2.0f), glm::vec3(0.0f, 0.0f, 0.0f), glm::vec3(0.0f, 1.0f, 0.0f));
         ubo.proj = perspective(radians(45.0f), swapChainExtent.width / (float)swapChainExtent.height, 0.1f, 10.0f);
         ubo.proj[1][1] *= -1; // flip the y-axis so that +y is up

         void* data;
         vkMapMemory(device, uniformBuffersMemory[currentImage], 0, sizeof(ubo), 0, &data);
         memcpy(data, &ubo, sizeof(ubo));
         vkUnmapMemory(device, uniformBuffersMemory[currentImage]);
      }

      void recreateSwapChain() {
         gui->refreshWindowSize();

         while (gui->getWindowWidth() == 0 || gui->getWindowHeight() == 0 ||
            (SDL_GetWindowFlags(window) & SDL_WINDOW_MINIMIZED) != 0) {
            SDL_WaitEvent(nullptr);
            gui->refreshWindowSize();
         }

         vkDeviceWaitIdle(device);

         cleanupSwapChain();

         createSwapChain();
         createImageViews();
         createRenderPass();

         createBufferResources();
      }

      void createBufferResources() {
         createDepthResources();
         createFramebuffers();
         createUniformBuffers();

         createGraphicsPipeline("shaders/scene-vert.spv", "shaders/scene-frag.spv", scenePipeline);
         createDescriptorPool(scenePipeline);
         createDescriptorSets(scenePipeline);

         createGraphicsPipeline("shaders/overlay-vert.spv", "shaders/overlay-frag.spv", overlayPipeline);
         createDescriptorPool(overlayPipeline);
         createDescriptorSets(overlayPipeline);

         createCommandBuffers();
      }

/*** START OF REFACTORED CODE ***/
      void cleanup() {
         cleanupSwapChain();
/*** END OF REFACTORED CODE ***/

         vkDestroySampler(device, textureSampler, nullptr);

         vkDestroyImageView(device, textureImageView, nullptr);
         vkDestroyImage(device, textureImage, nullptr);
         vkFreeMemory(device, textureImageMemory, nullptr);

         vkDestroyImageView(device, overlayImageView, nullptr);
         vkDestroyImage(device, overlayImage, nullptr);
         vkFreeMemory(device, overlayImageMemory, nullptr);

         vkDestroyImageView(device, sdlOverlayImageView, nullptr);
         vkDestroyImage(device, sdlOverlayImage, nullptr);
         vkFreeMemory(device, sdlOverlayImageMemory, nullptr);

         cleanupPipelineBuffers(scenePipeline);
         cleanupPipelineBuffers(overlayPipeline);

         for (size_t i = 0; i < MAX_FRAMES_IN_FLIGHT; i++) {
            vkDestroySemaphore(device, renderFinishedSemaphores[i], nullptr);
            vkDestroySemaphore(device, imageAvailableSemaphores[i], nullptr);
            vkDestroyFence(device, inFlightFences[i], nullptr);
         }

         vkDestroyCommandPool(device, commandPool, nullptr);
/*** START OF REFACTORED CODE ***/
         vkDestroyDevice(device, nullptr);
         vkDestroySurfaceKHR(instance, surface, nullptr);

         if (enableValidationLayers) {
            DestroyDebugUtilsMessengerEXT(instance, debugMessenger, nullptr);
         }

         vkDestroyInstance(instance, nullptr);
/*** END OF REFACTORED CODE ***/

         // TODO: Check if any of these functions accept null parameters
         // If they do, I don't need to check for that

         if (uiOverlay != nullptr) {
            SDL_DestroyTexture(uiOverlay);
            uiOverlay = nullptr;
         }

         TTF_CloseFont(gFont);
	      gFont = nullptr;

         if (uiText != nullptr) {
            SDL_DestroyTexture(uiText);
		      uiText = nullptr;
         }

         if (uiImage != nullptr) {
            SDL_DestroyTexture(uiImage);
            uiImage = nullptr;
         }

/*** START OF REFACTORED CODE ***/
         SDL_DestroyRenderer(gRenderer);
         gRenderer = nullptr;

         gui->destroyWindow();
         gui->shutdown();
         delete gui;
      }

      void cleanupSwapChain() {
/*** END OF REFACTORED CODE ***/
         vkDestroyImageView(device, depthImageView, nullptr);
         vkDestroyImage(device, depthImage, nullptr);
         vkFreeMemory(device, depthImageMemory, nullptr);

         for (auto framebuffer : swapChainFramebuffers) {
            vkDestroyFramebuffer(device, framebuffer, nullptr);
         }

         vkFreeCommandBuffers(device, commandPool, static_cast<uint32_t>(commandBuffers.size()), commandBuffers.data());

         cleanupPipeline(scenePipeline);
         cleanupPipeline(overlayPipeline);

         vkDestroyRenderPass(device, renderPass, nullptr);

/*** START OF REFACTORED CODE ***/
         for (auto imageView : swapChainImageViews) {
            vkDestroyImageView(device, imageView, nullptr);
         }

         vkDestroySwapchainKHR(device, swapChain, nullptr);
/*** END OF REFACTORED CODE ***/

         for (size_t i = 0; i < swapChainImages.size(); i++) {
            vkDestroyBuffer(device, uniformBuffers[i], nullptr);
            vkFreeMemory(device, uniformBuffersMemory[i], nullptr);
         }
/*** START OF REFACTORED CODE ***/
      }
/*** END OF REFACTORED CODE ***/

      void cleanupPipeline(GraphicsPipelineInfo& pipeline) {
         vkDestroyPipeline(device, pipeline.pipeline, nullptr);
         vkDestroyDescriptorPool(device, pipeline.descriptorPool, nullptr);
         vkDestroyPipelineLayout(device, pipeline.pipelineLayout, nullptr);
      }

      void cleanupPipelineBuffers(GraphicsPipelineInfo& pipeline) {
         vkDestroyDescriptorSetLayout(device, pipeline.descriptorSetLayout, nullptr);

         vkDestroyBuffer(device, pipeline.vertexBuffer, nullptr);
         vkFreeMemory(device, pipeline.vertexBufferMemory, nullptr);
         vkDestroyBuffer(device, pipeline.indexBuffer, nullptr);
         vkFreeMemory(device, pipeline.indexBufferMemory, nullptr);
      }

      static VKAPI_ATTR VkBool32 VKAPI_CALL debugCallback(
            VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity,
            VkDebugUtilsMessageTypeFlagsEXT messageType,
            const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData,
            void* pUserData) {
         cerr << "validation layer: " << pCallbackData->pMessage << endl;

         return VK_FALSE;
      }

      static vector<char> readFile(const string& filename) {
         ifstream file(filename, ios::ate | ios::binary);

         if (!file.is_open()) {
            throw runtime_error("failed to open file!");
         }

         size_t fileSize = (size_t) file.tellg();
         vector<char> buffer(fileSize);

         file.seekg(0);
         file.read(buffer.data(), fileSize);

         file.close();

         return buffer;
      }
};

/*** START OF REFACTORED CODE ***/
int main(int argc, char* argv[]) {

#ifdef NDEBUG
   cout << "DEBUGGING IS OFF" << endl;
#else
   cout << "DEBUGGING IS ON" << endl;
#endif

   cout << "Starting Vulkan game..." << endl;

   VulkanGame game;

   try {
      game.run();
   } catch (const exception& e) {
      cerr << e.what() << endl;
      return EXIT_FAILURE;
   }

   cout << "Finished running the game" << endl;

   return EXIT_SUCCESS;
}
/*** END OF REFACTORED CODE ***/
