#ifndef _VULKAN_GAME_H
#define _VULKAN_GAME_H

#include <glm/glm.hpp>

#include "game-gui-sdl.hpp"
#include "graphics-pipeline_vulkan.hpp"

#include "vulkan-utils.hpp"

#ifdef NDEBUG
   const bool ENABLE_VALIDATION_LAYERS = false;
#else
   const bool ENABLE_VALIDATION_LAYERS = true;
#endif

// TODO: Figure out if these structs should be defined in the VulkanGame class

struct Vertex {
   glm::vec3 pos;
   glm::vec3 color;
   glm::vec2 texCoord;
};

struct OverlayVertex {
   glm::vec3 pos;
   glm::vec2 texCoord;
};

class VulkanGame {
   public:
      VulkanGame(int maxFramesInFlight);
      ~VulkanGame();

      void run(int width, int height, unsigned char guiFlags);

   private:
      const int MAX_FRAMES_IN_FLIGHT;

      GameGui* gui;

      vector<GraphicsPipeline_Vulkan> graphicsPipelines;

      SDL_version sdlVersion;
      SDL_Window* window = nullptr;
      SDL_Renderer* renderer = nullptr;

      SDL_Texture* uiOverlay = nullptr;

      VkInstance instance;
      VkDebugUtilsMessengerEXT debugMessenger;
      VkSurfaceKHR surface; // TODO: Change the variable name to vulkanSurface
      VkPhysicalDevice physicalDevice = VK_NULL_HANDLE;
      VkDevice device;

      VkQueue graphicsQueue;
      VkQueue presentQueue;

      VkSwapchainKHR swapChain;
      vector<VkImage> swapChainImages;
      VkFormat swapChainImageFormat;
      VkExtent2D swapChainExtent;
      vector<VkImageView> swapChainImageViews;
      vector<VkFramebuffer> swapChainFramebuffers;

      VkRenderPass renderPass;
      VkCommandPool commandPool;
      vector<VkCommandBuffer> commandBuffers;

      VulkanImage depthImage;

      VkSampler textureSampler;

      vector<VkDescriptorBufferInfo> uniformBufferInfoList;

      // These are currently to store the MVP matrix
      // I should figure out if it makes sense to use them for other uniforms in the future
      // If not, I should rename them to better indicate their purpose.
      // I should also decide if I can use these for all shaders, or if I need a separapte set of buffers for each one
      vector<VkBuffer> uniformBuffers;
      vector<VkDeviceMemory> uniformBuffersMemory;

      VulkanImage floorTextureImage;
      VkDescriptorImageInfo floorTextureImageDescriptor;

      VulkanImage sdlOverlayImage;
      VkDescriptorImageInfo sdlOverlayImageDescriptor;

      vector<VkSemaphore> imageAvailableSemaphores;
      vector<VkSemaphore> renderFinishedSemaphores;
      vector<VkFence> inFlightFences;

      size_t currentFrame;

      bool framebufferResized;

      bool initWindow(int width, int height, unsigned char guiFlags);
      void initVulkan();
      void mainLoop();
      void renderUI();
      void renderScene();
      void cleanup();

      void createVulkanInstance(const vector<const char*> &validationLayers);
      void setupDebugMessenger();
      void populateDebugMessengerCreateInfo(VkDebugUtilsMessengerCreateInfoEXT& createInfo);
      void createVulkanSurface();
      void pickPhysicalDevice(const vector<const char*>& deviceExtensions);
      bool isDeviceSuitable(VkPhysicalDevice physicalDevice, const vector<const char*>& deviceExtensions);
      void createLogicalDevice(
         const vector<const char*> validationLayers,
         const vector<const char*>& deviceExtensions);
      void createSwapChain();
      void createImageViews();
      void createRenderPass();
      VkFormat findDepthFormat();
      void createCommandPool();
      void createImageResources();

      void createTextureSampler();
      void createFramebuffers();
      void createUniformBuffers();
      void createCommandBuffers();
      void createSyncObjects();

      void recreateSwapChain();
      void updateUniformBuffer(uint32_t currentImage);

      void cleanupSwapChain();

      static VKAPI_ATTR VkBool32 VKAPI_CALL debugCallback(
            VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity,
            VkDebugUtilsMessageTypeFlagsEXT messageType,
            const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData,
            void* pUserData);
};

#endif // _VULKAN_GAME_H