#include "logger.h"

#include "stb_image.h"

#define GLM_SWIZZLE
#include <glm/mat4x4.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>

#include <GL/glew.h>
#include <GLFW/glfw3.h>

#include <cstdio>
#include <iostream>
#include <fstream>

#define _USE_MATH_DEFINES
#include <cmath>

using namespace std;
using namespace glm;

#define ONE_DEG_IN_RAD (2.0 * M_PI) / 360.0 // 0.017444444

const bool FULLSCREEN = false;
int width = 640;
int height = 480;

vec3 cam_pos;

vec3 face_point1, face_point2, face_point3;

mat4 view_mat;
mat4 proj_mat;

GLuint loadShader(GLenum type, string file);
GLuint loadShaderProgram(string vertexShaderPath, string fragmentShaderPath);
unsigned char* loadImage(string file_name, int* x, int* y);

void glfw_error_callback(int error, const char* description) {
   gl_log_err("GLFW ERROR: code %i msg: %s\n", error, description);
}

void mouse_button_callback(GLFWwindow* window, int button, int action, int mods) {
   double mouse_x, mouse_y;
   glfwGetCursorPos(window, &mouse_x, &mouse_y);

   if (button == GLFW_MOUSE_BUTTON_LEFT && action == GLFW_PRESS) {
      cout << "Mouse clicked (" << mouse_x << "," << mouse_y << ")" << endl;

      float x = (2.0f*mouse_x) / width - 1.0f;
      float y = 1.0f - (2.0f*mouse_y) / height;

      vec4 ray_clip = vec4(x, y, -1.0f, 1.0f);
      vec4 ray_eye = inverse(proj_mat) * ray_clip;
      ray_eye = vec4(ray_eye.xy(), -1.0f, 0.0f);
      vec3 ray_world = normalize((inverse(view_mat) * ray_eye).xyz());

      vec3 click_point = cam_pos + ray_world;

      /* Now, we need to generate the constants for the equations describing
       * a 3D line:
       *   (x - x0) / a = (y - y0) / b = (z - z0) / c
       *
       * The line goes through the camera position, so
       * cam_pos = <x0, y0, z0>
       */

      cout << "Converted -> (" << ray_world.x << "," << ray_world.y << "," << ray_world.z << ")" << endl << endl;;
      cout << "Camera -> (" << cam_pos.x << "," << cam_pos.y << "," << cam_pos.z << ")" << endl;
      cout << "Click point -> (" << click_point.x << "," << click_point.y << "," << click_point.z << ")" << endl;

      float a = 1.0f;
      float b = a * (click_point.y - cam_pos.y) / (click_point.x - cam_pos.x);
      float c = a * (click_point.z - cam_pos.z) / (click_point.x - cam_pos.x);

      cout << "(x - " << cam_pos.x << ") / " << a << " = ";
      cout << "(y - " << cam_pos.y << ") / " << b << " = ";
      cout << "(z - " << cam_pos.z << ") / " << c << endl;;

      /* Now, we need to generate the constants for the equations describing
       * a 3D plane:
       * dx + ey +fz +g = 0
       */

      cout << "Points on the plane" << endl;
      cout << "(" << face_point1.x << "," << face_point1.y << "," << face_point1.z << ")" << endl;
      cout << "(" << face_point2.x << "," << face_point2.y << "," << face_point2.z << ")" << endl;
      cout << "(" << face_point3.x << "," << face_point3.y << "," << face_point3.z << ")" << endl;

      // get intersection
   }
}

int main(int argc, char* argv[]) {
   cout << "New OpenGL Game" << endl;

   if (!restart_gl_log()) {}
   gl_log("starting GLFW\n%s\n", glfwGetVersionString());

   glfwSetErrorCallback(glfw_error_callback);
   if (!glfwInit()) {
      fprintf(stderr, "ERROR: could not start GLFW3\n");
      return 1;
   }

#ifdef __APPLE__
   glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
   glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
   glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
   glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
#endif

   glfwWindowHint(GLFW_SAMPLES, 4);

   GLFWwindow* window = NULL;

   if (FULLSCREEN) {
      GLFWmonitor* mon = glfwGetPrimaryMonitor();
      const GLFWvidmode* vmode = glfwGetVideoMode(mon);

      cout << "Fullscreen resolution " << vmode->width << "x" << vmode->height << endl;
      window = glfwCreateWindow(vmode->width, vmode->height, "Extended GL Init", mon, NULL);

      width = vmode->width;
      height = vmode->height;
   } else {
      window = glfwCreateWindow(width, height, "Hello Triangle", NULL, NULL);
   }

   if (!window) {
      fprintf(stderr, "ERROR: could not open window with GLFW3\n");
      glfwTerminate();
      return 1;
   }

   glfwSetMouseButtonCallback(window, mouse_button_callback);

   glfwMakeContextCurrent(window);
   glewExperimental = GL_TRUE;
   glewInit();

   // glViewport(0, 0, width*2, height*2);

   const GLubyte* renderer = glGetString(GL_RENDERER);
   const GLubyte* version = glGetString(GL_VERSION);
   printf("Renderer: %s\n", renderer);
   printf("OpenGL version supported %s\n", version);

   glEnable(GL_DEPTH_TEST);
   glDepthFunc(GL_LESS);

   glEnable(GL_CULL_FACE);
   // glCullFace(GL_BACK);
   // glFrontFace(GL_CW);

   int x, y;
   unsigned char* texImage = loadImage("test.png", &x, &y);
   if (texImage) {
     cout << "Yay, I loaded an image!" << endl;
     cout << x << endl;
     cout << y << endl;
     printf ("first 4 bytes are: %i %i %i %i\n", texImage[0], texImage[1], texImage[2], texImage[3]);
   }

   GLuint tex = 0;
   glGenTextures(1, &tex);
   glActiveTexture(GL_TEXTURE0);
   glBindTexture(GL_TEXTURE_2D, tex);
   glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, x, y, 0, GL_RGBA, GL_UNSIGNED_BYTE, texImage);

   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

   GLfloat points[] = {
      0.0f,  0.5f,  0.0f,
     -0.5f, -0.5f,  0.0f,
      0.5f, -0.5f,  0.0f,
      0.5f, -0.5f,  0.0f,
     -0.5f, -0.5f,  0.0f,
      0.0f,  0.5f,  0.0f,
   };

   // initialize global variables for click intersection test
   face_point1 = vec3(points[0], points[1], points[2]);
   face_point2 = vec3(points[3], points[4], points[5]);
   face_point3 = vec3(points[6], points[7], points[8]);

   GLfloat colors[] = {
     1.0, 0.0, 0.0,
     0.0, 0.0, 1.0,
     0.0, 1.0, 0.0,
     0.0, 1.0, 0.0,
     0.0, 0.0, 1.0,
     1.0, 0.0, 0.0,
   };

   // Each point is made of 3 floats
   int numPoints = (sizeof(points) / sizeof(float)) / 3;

   GLfloat points2[] = {
      0.5f,  0.5f,  0.0f,
     -0.5f,  0.5f,  0.0f,
     -0.5f, -0.5f,  0.0f,
      0.5f,  0.5f,  0.0f,
     -0.5f, -0.5f,  0.0f,
      0.5f, -0.5f,  0.0f,
   };

   /*
   GLfloat colors2[] = {
     0.0, 0.9, 0.9,
     0.0, 0.9, 0.9,
     0.0, 0.9, 0.9,
     0.0, 0.9, 0.9,
     0.0, 0.9, 0.9,
     0.0, 0.9, 0.9,
   };
   */

   GLfloat texcoords[] = {
     1.0f, 1.0f,
     0.0f, 1.0f,
     0.0, 0.0,
     1.0, 1.0,
     0.0, 0.0,
     1.0, 0.0
   };

   // Each point is made of 3 floats
   int numPoints2 = (sizeof(points2) / sizeof(float)) / 3;

   /*
   mat4 T_model = translate(mat4(), vec3(0.5f, 0.0f, 0.0f));
   mat4 R_model = rotate(mat4(), 4.0f, vec3(0.0f, 1.0f, 0.0f));
   */
   mat4 T_model = translate(mat4(), vec3(0.0f, 0.0f, 0.0f));
   mat4 R_model = rotate(mat4(), 0.0f, vec3(0.0f, 1.0f, 0.0f));
   mat4 model_mat = T_model*R_model;

   mat4 T_model2 = translate(mat4(), vec3(-1.0f, 0.0f, 0.0f));
   mat4 R_model2 = rotate(mat4(), 0.0f, vec3(0.0f, 1.0f, 0.0f));
   mat4 model_mat2 = T_model2*R_model2;

   GLuint points_vbo = 0;
   glGenBuffers(1, &points_vbo);
   glBindBuffer(GL_ARRAY_BUFFER, points_vbo);
   glBufferData(GL_ARRAY_BUFFER, sizeof(points), points, GL_STATIC_DRAW);

   GLuint colors_vbo = 0;
   glGenBuffers(1, &colors_vbo);
   glBindBuffer(GL_ARRAY_BUFFER, colors_vbo);
   glBufferData(GL_ARRAY_BUFFER, sizeof(colors), colors, GL_STATIC_DRAW);

   GLuint vao = 0;
   glGenVertexArrays(1, &vao);
   glBindVertexArray(vao);
   glBindBuffer(GL_ARRAY_BUFFER, points_vbo);
   glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, NULL);
   glBindBuffer(GL_ARRAY_BUFFER, colors_vbo);
   glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, NULL);

   glEnableVertexAttribArray(0);
   glEnableVertexAttribArray(1);

   GLuint points2_vbo = 0;
   glGenBuffers(1, &points2_vbo);
   glBindBuffer(GL_ARRAY_BUFFER, points2_vbo);
   glBufferData(GL_ARRAY_BUFFER, sizeof(points2), points2, GL_STATIC_DRAW);

   /*
   GLuint colors2_vbo = 0;
   glGenBuffers(1, &colors2_vbo);
   glBindBuffer(GL_ARRAY_BUFFER, colors2_vbo);
   glBufferData(GL_ARRAY_BUFFER, sizeof(colors2), colors2, GL_STATIC_DRAW);
   */

   GLuint vt_vbo;
   glGenBuffers(1, &vt_vbo);
   glBindBuffer(GL_ARRAY_BUFFER, vt_vbo);
   glBufferData(GL_ARRAY_BUFFER, sizeof(texcoords), texcoords, GL_STATIC_DRAW);

   GLuint vao2 = 0;
   glGenVertexArrays(1, &vao2);
   glBindVertexArray(vao2);
   glBindBuffer(GL_ARRAY_BUFFER, points2_vbo);
   glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, NULL);
   // glBindBuffer(GL_ARRAY_BUFFER, colors2_vbo);
   // glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, NULL);
   glBindBuffer(GL_ARRAY_BUFFER, vt_vbo);
   glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, NULL);

   glEnableVertexAttribArray(0);
   glEnableVertexAttribArray(1);

   GLuint shader_program = loadShaderProgram("./color.vert", "./color.frag");
   GLuint shader_program2 = loadShaderProgram("./texture.vert", "./texture.frag");

   float speed = 1.0f;
   float last_position = 0.0f;

   float cam_speed = 1.0f;
   float cam_yaw_speed = 60.0f*ONE_DEG_IN_RAD;

   cam_pos = vec3(0.0f, 0.0f, 2.0f);
   float cam_yaw = 0.0f;

   mat4 T = translate(mat4(), vec3(-cam_pos.x, -cam_pos.y, -cam_pos.z));
   mat4 R = rotate(mat4(), -cam_yaw, vec3(0.0f, 1.0f, 0.0f));
   view_mat = R*T;

   float near = 0.1f;
   float far = 100.0f;
   float fov = 67.0f * ONE_DEG_IN_RAD;
   float aspect = (float)width / (float)height;

   float range = tan(fov * 0.5f) * near;
   float Sx = near / (range * aspect);
   float Sy = near / range;
   float Sz = -(far + near) / (far - near);
   float Pz = -(2.0f * far * near) / (far - near);

   float proj_arr[] = {
     Sx, 0.0f, 0.0f, 0.0f,
     0.0f, Sy, 0.0f, 0.0f,
     0.0f, 0.0f, Sz, -1.0f,
     0.0f, 0.0f, Pz, 0.0f,
   };
   proj_mat = make_mat4(proj_arr);

   GLint model_mat_loc = glGetUniformLocation(shader_program2, "model");
   GLint view_mat_loc = glGetUniformLocation(shader_program2, "view");
   GLint proj_mat_loc = glGetUniformLocation(shader_program2, "proj");

   GLint model_test_loc = glGetUniformLocation(shader_program, "model");
   GLint view_test_loc = glGetUniformLocation(shader_program, "view");
   GLint proj_test_loc = glGetUniformLocation(shader_program, "proj");

   glUseProgram(shader_program);
   glUniformMatrix4fv(model_test_loc, 1, GL_FALSE, value_ptr(model_mat));
   glUniformMatrix4fv(proj_test_loc, 1, GL_FALSE, value_ptr(proj_mat));

   glUseProgram(shader_program2);
   glUniformMatrix4fv(model_mat_loc, 1, GL_FALSE, value_ptr(model_mat2));
   glUniformMatrix4fv(proj_mat_loc, 1, GL_FALSE, value_ptr(proj_mat));

   // glUniform1i(tex_loc, 0);

   bool cam_moved = false;

   double previous_seconds = glfwGetTime();
   while (!glfwWindowShouldClose(window)) {
      double current_seconds = glfwGetTime();
      double elapsed_seconds = current_seconds - previous_seconds;
      previous_seconds = current_seconds;

      if (fabs(last_position) > 1.0f) {
         speed = -speed;
      }

      /*
      model[12] = last_position + speed*elapsed_seconds;
      last_position = model[12];
      */

      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

      glUseProgram(shader_program);
      glUniformMatrix4fv(view_test_loc, 1, GL_FALSE, value_ptr(view_mat));

      glBindVertexArray(vao);

      glDrawArrays(GL_TRIANGLES, 0, numPoints);

      glUseProgram(shader_program2);
      glUniformMatrix4fv(view_mat_loc, 1, GL_FALSE, value_ptr(view_mat));

      glBindVertexArray(vao2);

      glDrawArrays(GL_TRIANGLES, 0, numPoints2);

      glfwPollEvents();
      glfwSwapBuffers(window);

      if (GLFW_PRESS == glfwGetKey(window, GLFW_KEY_ESCAPE)) {
         glfwSetWindowShouldClose(window, 1);
      }

      float dist = cam_speed * elapsed_seconds;
      if (glfwGetKey(window, GLFW_KEY_A)) {
         cam_pos.x -= cos(cam_yaw)*dist;
         cam_pos.z += sin(cam_yaw)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_D)) {
         cam_pos.x += cos(cam_yaw)*dist;
         cam_pos.z -= sin(cam_yaw)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_W)) {
         cam_pos.x -= sin(cam_yaw)*dist;
         cam_pos.z -= cos(cam_yaw)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_S)) {
         cam_pos.x += sin(cam_yaw)*dist;
         cam_pos.z += cos(cam_yaw)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_LEFT)) {
         cam_yaw += cam_yaw_speed * elapsed_seconds;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_RIGHT)) {
         cam_yaw -= cam_yaw_speed * elapsed_seconds;
         cam_moved = true;
      }
      if (cam_moved) {
         T = translate(mat4(), vec3(-cam_pos.x, -cam_pos.y, -cam_pos.z));
         R = rotate(mat4(), -cam_yaw, vec3(0.0f, 1.0f, 0.0f));
         view_mat = R*T;

         glUniformMatrix4fv(view_mat_loc, 1, GL_FALSE, value_ptr(view_mat));
         cam_moved = false;
      }
   }

   glfwTerminate();
   return 0;
}

GLuint loadShader(GLenum type, string file) {
  cout << "Loading shader from file " << file << endl;

  ifstream shaderFile(file);
  GLuint shaderId = 0;

  if (shaderFile.is_open()) {
    string line, shaderString;

    while(getline(shaderFile, line)) {
      shaderString += line + "\n";
    }
    shaderFile.close();
    const char* shaderCString = shaderString.c_str();

    shaderId = glCreateShader(type);
    glShaderSource(shaderId, 1, &shaderCString, NULL);
    glCompileShader(shaderId);

    cout << "Loaded successfully" << endl;
  } else {
    cout << "Failed to loade the file" << endl;
  }

  return shaderId;
}

GLuint loadShaderProgram(string vertexShaderPath, string fragmentShaderPath) {
   GLuint vs = loadShader(GL_VERTEX_SHADER, vertexShaderPath);
   GLuint fs = loadShader(GL_FRAGMENT_SHADER, fragmentShaderPath);

   GLuint shader_program = glCreateProgram();
   glAttachShader(shader_program, vs);
   glAttachShader(shader_program, fs);

   glLinkProgram(shader_program);

   return shader_program;
}

unsigned char* loadImage(string file_name, int* x, int* y) {
  int n;
  int force_channels = 4;
  unsigned char* image_data = stbi_load(file_name.c_str(), x, y, &n, force_channels);
  if (!image_data) {
    fprintf(stderr, "ERROR: could not load %s\n", file_name.c_str());
  }
  return image_data;
}
