#include "game-gui-sdl.hpp"

#include <map>
#include <queue>

#include <SDL2/SDL_ttf.h>

#include "compiler.hpp"
#include "consts.hpp"

using namespace std;

GameGui_SDL::GameGui_SDL() : keyState(SDL_GetKeyboardState(NULL)) {
   window = nullptr;
}

string& GameGui_SDL::getError() {
   s_errorMessage = SDL_GetError();

   return s_errorMessage;
}

bool GameGui_SDL::init() {
   // may want to define SDL_INIT_NOPARACHUTE when I start handling crashes since it
   // prevents SDL from setting up its own handlers for SIG_SEGV and stuff like that

   s_errorMessage = "No error";

   if (SDL_Init(SDL_INIT_EVERYTHING) < 0) {
      return RTWO_ERROR;
   }

   if (TTF_Init() == -1) {
      return RTWO_ERROR;
   }

   return RTWO_SUCCESS;
}

void GameGui_SDL::shutdown() {
   SDL_Quit();
}

void* GameGui_SDL::createWindow(const string& title, int width, int height, bool fullscreen) {
   // TODO: Make an OpenGL version of the SDL_CreateWindow call as well

   // On Apple's OS X you must set the NSHighResolutionCapable Info.plist property to YES,
   // otherwise you will not receive a High DPI OpenGL canvas.

   SDL_DisplayMode dm;
   SDL_GetCurrentDisplayMode(0, &dm);

   if (fullscreen) {
      width = dm.w;
      height = dm.h;
   }

#ifdef WINDOWS
   uint32_t flags = SDL_WINDOW_VULKAN | SDL_WINDOW_SHOWN | SDL_WINDOW_ALLOW_HIGHDPI |
      (fullscreen ? SDL_WINDOW_FULLSCREEN_DESKTOP : SDL_WINDOW_RESIZABLE);
#else
   uint32_t flags = SDL_WINDOW_VULKAN | SDL_WINDOW_SHOWN | SDL_WINDOW_ALLOW_HIGHDPI |
      (fullscreen ? SDL_WINDOW_FULLSCREEN : SDL_WINDOW_RESIZABLE);
#endif

   window = SDL_CreateWindow(title.c_str(), SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, width, height, flags);

   refreshWindowSize();

   return window;
}

void GameGui_SDL::destroyWindow() {
   // TODO: This function can throw some errors. They should be handled
   SDL_DestroyWindow(window);
}

void GameGui_SDL::processEvents() {
}

int GameGui_SDL::pollEvent(UIEvent* event) {
   SDL_Event e;

   /* The trackpad on OSX triggers both SDL_MOUSE and SDL_FINGER events, so just treat them both
    * as mouse events since this game isn't targeting mobile devices
    */

   if (SDL_PollEvent(&e)) {
      switch(e.type) {
         case SDL_QUIT:
            event->type = UI_EVENT_QUIT;
            break;
         case SDL_WINDOWEVENT:
            if (e.window.event == SDL_WINDOWEVENT_SIZE_CHANGED ||
               e.window.event == SDL_WINDOWEVENT_MINIMIZED ||
               e.window.event == SDL_WINDOWEVENT_MAXIMIZED) {
               event->type = UI_EVENT_WINDOWRESIZE;
            } else {
               event->type = UI_EVENT_WINDOW;
            }
            break;
         case SDL_KEYDOWN:
            event->type = UI_EVENT_KEYDOWN;
            event->key.keycode = e.key.keysym.scancode;
            event->key.repeat = e.key.repeat != 0;
            break;
         case SDL_KEYUP:
            event->type = UI_EVENT_KEYUP;
            event->key.keycode = e.key.keysym.scancode;
            event->key.repeat = e.key.repeat != 0;
            break;
         case SDL_MOUSEBUTTONDOWN:
         case SDL_FINGERDOWN:
            event->type = UI_EVENT_MOUSEBUTTONDOWN;
            break;
         case SDL_MOUSEBUTTONUP:
         case SDL_FINGERUP:
            event->type = UI_EVENT_MOUSEBUTTONUP;
            break;
         case SDL_MOUSEMOTION:
            event->mouse.x = e.motion.x;
            event->mouse.y = e.motion.y;
         case SDL_FINGERMOTION:
            event->type = UI_EVENT_MOUSEMOTION;
            break;
         // Ignore the following events
         case SDL_AUDIODEVICEADDED:
         case SDL_AUDIODEVICEREMOVED:
         case SDL_TEXTINPUT:
         case SDL_TEXTEDITING:
            event->type = UI_EVENT_UNKNOWN;
            event->unknown.eventType = e.type;
            break;
         default:
            event->type = UI_EVENT_UNKNOWN;
            event->unknown.eventType = 0;
      }

      return 1;
   }

   event = nullptr;
   return 0;
}

bool GameGui_SDL::keyPressed(unsigned int key) {
   return keyState[key];
}

void GameGui_SDL::refreshWindowSize() {
   // TODO: Make sure this works on a mac (the analogous glfw function had issues on Mac retina displays)
   SDL_GetWindowSize(window, &windowWidth, &windowHeight);
}

int GameGui_SDL::getWindowWidth() {
   return windowWidth;
}

int GameGui_SDL::getWindowHeight() {
   return windowHeight;
}

#ifdef GAMEGUI_INCLUDE_VULKAN

bool GameGui_SDL::createVulkanSurface(VkInstance instance, VkSurfaceKHR* surface) {
   return SDL_Vulkan_CreateSurface(window, instance, surface) ? RTWO_SUCCESS : RTWO_ERROR;
}

vector<const char*> GameGui_SDL::getRequiredExtensions() {
   uint32_t extensionCount = 0;
   SDL_Vulkan_GetInstanceExtensions(window, &extensionCount, nullptr);

   vector<const char*> extensions(extensionCount);
   SDL_Vulkan_GetInstanceExtensions(window, &extensionCount, extensions.data());

   return extensions;
}

#endif
