#include "logger.h"

#include <GL/glew.h>
#include <GLFW/glfw3.h>

#include <cstdio>
#include <iostream>
#include <fstream>
#include <cmath>

using namespace std;

GLuint loadShader(GLenum type, string file);
GLuint createDataBuffer(size_t size, GLfloat* data);
GLuint createArrayBuffer(GLuint points_vbo, GLuint colors_vbo);

GLfloat* createCirclePoints(unsigned int smoothness, GLfloat centerX, GLfloat centerY, GLfloat radius);
GLfloat* createCircleColors(unsigned int smoothness);

const double PI = 3.1415926535897;
const bool FULLSCREEN = false;

void glfw_error_callback(int error, const char* description) {
   gl_log_err("GLFW ERROR: code %i msg: %s\n", error, description);
}

int main(int argc, char* argv[]) {
   cout << "New OpenGL Game" << endl;

   if (!restart_gl_log()) {}
   gl_log("starting GLFW\n%s\n", glfwGetVersionString());

   glfwSetErrorCallback(glfw_error_callback);
   if (!glfwInit()) {
      fprintf(stderr, "ERROR: could not start GLFW3\n");
      return 1;
   }

#ifdef __APPLE__
   glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
   glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
   glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
   glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
#endif

   glfwWindowHint(GLFW_SAMPLES, 4);

   GLFWwindow* window = NULL;

   int width = 640;
   int height = 480;

   if (FULLSCREEN) {
      GLFWmonitor* mon = glfwGetPrimaryMonitor();
      const GLFWvidmode* vmode = glfwGetVideoMode(mon);

      cout << "Fullscreen resolution " << vmode->width << "x" << vmode->height << endl;
      window = glfwCreateWindow(vmode->width, vmode->height, "Extended GL Init", mon, NULL);

      width = vmode->width;
      height = vmode->height;
   } else {
      window = glfwCreateWindow(width, height, "Hello Triangle", NULL, NULL);
   }

   if (!window) {
      fprintf(stderr, "ERROR: could not open window with GLFW3\n");
      glfwTerminate();
      return 1;
   }
   glfwMakeContextCurrent(window);
   glewExperimental = GL_TRUE;
   glewInit();

   // glViewport(0, 0, width*2, height*2);

   const GLubyte* renderer = glGetString(GL_RENDERER);
   const GLubyte* version = glGetString(GL_VERSION);
   printf("Renderer: %s\n", renderer);
   printf("OpenGL version supported %s\n", version);

   glEnable(GL_DEPTH_TEST);
   glDepthFunc(GL_LESS);

   glEnable(GL_CULL_FACE);
   // glCullFace(GL_BACK);
   // glFrontFace(GL_CW);

   unsigned int numBallPoints = 36;

   GLfloat ballRadius = 0.2f;
   GLfloat paddleThickness = 0.1f;

   GLfloat* points = createCirclePoints(numBallPoints, 0.0f, 0.0f, ballRadius);
   GLfloat* colors = createCircleColors(numBallPoints);

   GLfloat points_paddle[] = {
      -1.0f,  0.15f,  0.0f,
      -1.0f, -0.15f,  0.0f,
      -0.9f, -0.15f,  0.0f,
      -1.0f,  0.15f,  0.0f,
      -0.9f, -0.15f,  0.0f,
      -0.9f,  0.15f,  0.0f,
   };

   GLfloat colors_paddle[] = {
      1.0, 0.0, 0.0,
      1.0, 0.0, 0.0,
      1.0, 0.0, 0.0,
      1.0, 0.0, 0.0,
      1.0, 0.0, 0.0,
      1.0, 0.0, 0.0,
   };

   GLfloat model[] = {
     1.0f, 0.0f, 0.0f, 0.0f, // column 1
     0.0f, 1.0f, 0.0f, 0.0f, // column 2
     0.0f, 0.0f, 1.0f, 0.0f, // column 3
     0.0f, 0.0f, 0.0f, 1.0f, // column 4
   };

   GLuint ball_points_vbo = createDataBuffer(numBallPoints*9*sizeof(GLfloat), points);
   GLuint ball_colors_vbo = createDataBuffer(numBallPoints*9*sizeof(GLfloat), colors);
   GLuint ball_vao = createArrayBuffer(ball_points_vbo, ball_colors_vbo);

   GLuint paddle_points_vbo = createDataBuffer(sizeof(points_paddle), points_paddle);
   GLuint paddle_colors_vbo = createDataBuffer(sizeof(colors_paddle), colors_paddle);
   GLuint paddle_vao = createArrayBuffer(paddle_points_vbo, paddle_colors_vbo);

   GLuint vs = loadShader(GL_VERTEX_SHADER, "./test.vert");
   GLuint fs = loadShader(GL_FRAGMENT_SHADER, "./test.frag");

   GLuint shader_program = glCreateProgram();
   glAttachShader(shader_program, vs);
   glAttachShader(shader_program, fs);

   glLinkProgram(shader_program);
   glUseProgram(shader_program);

   GLint location = glGetUniformLocation(shader_program, "model");

   float speedX = 1.0f;
   float speedY = 0.0f;
   float last_position_x = 0.0f;
   float last_position_y = 0.0f;
   float last_paddle_pos = 0.0f;

   double previous_seconds = glfwGetTime();
   while (!glfwWindowShouldClose(window)) {
      double current_seconds = glfwGetTime();
      double elapsed_seconds = current_seconds - previous_seconds;
      previous_seconds = current_seconds;

      if (last_position_x > 1.0f-ballRadius) {
         speedX = -speedX;
      }
      if (last_position_y+ballRadius>1.0f || last_position_y-ballRadius<-1.0f) {
         speedY = -speedY;
      }

      if (last_position_x < -1.0f+ballRadius+paddleThickness &&
          last_position_x > -1.0f) {
         if (fabs(last_paddle_pos-last_position_y) < 0.15f) {
            speedX = -speedX;
         } else {
            // if the ball hits the paddle on a corner, make it bounce off
            // at an angle
            float horDist = fabs(paddleThickness-1.0f-last_position_x);
            float verDist1 = fabs(last_paddle_pos+0.15f-last_position_y);
            float verDist2 = fabs(last_paddle_pos-0.15f-last_position_y);

            float dist1 = pow(horDist, 2)+pow(verDist1, 2);
            float dist2 = pow(horDist, 2)+pow(verDist2, 2);

            if (dist1 < ballRadius) {
               if (speedX == -1.0f || speedY == 0.7f) {
                  speedX = 0.7f;
                  speedY = 0.7f;
               } else {
                  speedX = 1.0f;
                  speedY = 0.0f;
               }
            } else if (dist2 < ballRadius) {
               if (speedX == -1.0f || speedY == -0.7f) {
                  speedX = 0.7f;
                  speedY = -0.7f;
               } else {
                  speedX = 1.0f;
                  speedY = 0.0f;
               }
            }
         }
      }

      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

      model[12] = 0.0f;
      model[13] = last_paddle_pos;
      glUniformMatrix4fv(location, 1, GL_FALSE, model);

      glBindVertexArray(paddle_vao);
      glEnableVertexAttribArray(0);
      glEnableVertexAttribArray(1);

      glDrawArrays(GL_TRIANGLES, 0, sizeof(points_paddle)/sizeof(GLfloat)/3);

      model[12] = last_position_x + speedX*elapsed_seconds;
      last_position_x = model[12];
      model[13] = last_position_y + speedY*elapsed_seconds;
      last_position_y = model[13];
      glUniformMatrix4fv(location, 1, GL_FALSE, model);

      glBindVertexArray(ball_vao);
      glEnableVertexAttribArray(0);
      glEnableVertexAttribArray(1);

      glDrawArrays(GL_TRIANGLES, 0, numBallPoints*3);

      glfwPollEvents();
      glfwSwapBuffers(window);

      if (GLFW_PRESS == glfwGetKey(window, GLFW_KEY_ESCAPE)) {
         glfwSetWindowShouldClose(window, 1);
      }
      if (GLFW_PRESS == glfwGetKey(window, GLFW_KEY_UP)) {
         last_paddle_pos += 1.0f * elapsed_seconds;
         if (last_paddle_pos > 0.85f) {
            last_paddle_pos = 0.85f;
         }
      }
      if (GLFW_PRESS == glfwGetKey(window, GLFW_KEY_DOWN)) {
         last_paddle_pos -= 1.0f * elapsed_seconds;
         if (last_paddle_pos < -0.85f) {
            last_paddle_pos = -0.85f;
         }
      }
   }

   glfwTerminate();

   delete points;
   delete colors;

   return 0;
}

GLuint loadShader(GLenum type, string file) {
  cout << "Loading shader from file " << file << endl;

  ifstream shaderFile(file);
  GLuint shaderId = 0;

  if (shaderFile.is_open()) {
    string line, shaderString;

    while(getline(shaderFile, line)) {
      shaderString += line + "\n";
    }
    shaderFile.close();
    const char* shaderCString = shaderString.c_str();

    shaderId = glCreateShader(type);
    glShaderSource(shaderId, 1, &shaderCString, NULL);
    glCompileShader(shaderId);

    cout << "Loaded successfully" << endl;
  } else {
    cout << "Failed to loade the file" << endl;
  }

  return shaderId;
}

GLuint createDataBuffer(size_t size, GLfloat* data) {
   GLuint vbo = 0;
   glGenBuffers(1, &vbo);
   glBindBuffer(GL_ARRAY_BUFFER, vbo);
   glBufferData(GL_ARRAY_BUFFER, size, data, GL_STATIC_DRAW);
   return vbo;
}

GLuint createArrayBuffer(GLuint points_vbo, GLuint colors_vbo) {
   GLuint vao = 0;
   glGenVertexArrays(1, &vao);
   glBindVertexArray(vao);
   glBindBuffer(GL_ARRAY_BUFFER, points_vbo);
   glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, NULL);
   glBindBuffer(GL_ARRAY_BUFFER, colors_vbo);
   glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, NULL);
   return vao;
}

GLfloat* createCirclePoints(unsigned int smoothness, GLfloat centerX, GLfloat centerY, GLfloat radius) {
   GLfloat* points = new GLfloat[smoothness*9];

   GLfloat curX, curY, nextX, nextY;
   curX = centerX;
   curY = centerY+radius;

   for (unsigned int i=0; i<smoothness; i++) {
      double radians = PI/2 + 2*PI * (i+1)/smoothness;
      nextX = centerX + cos(radians)*radius;
      nextY = centerY + sin(radians)*radius;

      points[i*9] = curX;
      points[i*9+1] = curY;
      points[i*9+2] = 0.5f;
      points[i*9+3] = nextX;
      points[i*9+4] = nextY;
      points[i*9+5] = 0.5f;
      points[i*9+6] = centerX;
      points[i*9+7] = centerY;
      points[i*9+8] = 0.5f;

      curX = nextX;
      curY = nextY;
   }

   return points;
}

GLfloat* createCircleColors(unsigned int smoothness) {
   GLfloat* colors = new GLfloat[smoothness*9];

   for (unsigned int i=0; i<smoothness*3; i++) {
      colors[i*3] = 0.0f;
      colors[i*3+1] = 1.0f;
      colors[i*3+2] = 0.0f;
   }

   return colors;
}
