package main;

import java.awt.image.BufferedImage;
import java.util.*;

public class Weapon extends Item {

  BufferedImage[] imgProj;

  protected LinkedList<Effect> spawnEffects;  
  protected LinkedList<Effect> contactEffects;

  private int attackSpeed;
  private int range;

  public Weapon(String name, ItemType type, String img, BufferedImage[] imgProj, int attackSpeed, int range, int damage) {
    super(name, type, "weapons/" + img);
    this.imgProj = imgProj;
    this.spawnEffects = new LinkedList<Effect>();
    this.contactEffects = new LinkedList<Effect>();
    this.contactEffects.add(new Damage(damage, TargetType.Enemy));
    this.attackSpeed = attackSpeed;
    this.range = range;
  }

  public Weapon(Weapon copy, Point loc) {
    super(copy, loc);
    this.imgProj = copy.imgProj;
    this.spawnEffects = new LinkedList<Effect>();
    Iterator<Effect> iter = copy.spawnEffects.iterator();
    while (iter.hasNext()) {
      this.spawnEffects.add(((Effect)iter.next()).copy());
    }
    this.contactEffects = new LinkedList<Effect>();
    iter = copy.contactEffects.iterator();
    while (iter.hasNext()) {
      this.contactEffects.add(((Effect)iter.next()).copy());
    }
    this.attackSpeed = copy.attackSpeed;
    this.range = copy.range;
  }

  public Weapon copy(Point loc) {
    return new Weapon(this, loc);
  }

  public void addSpawnEffect(Effect effect) {
    this.spawnEffects.add(effect);
  }

  public void applySpawnEffects(Creature cr) {
    Iterator<Effect> iter = this.spawnEffects.iterator();
    while (iter.hasNext()) {
      ((Effect)iter.next()).applyEffect(cr);
    } 
  }

  public void addContactEffect(Effect effect) {
    this.contactEffects.add(effect);
  }

  public void applyContactEffects(Creature cr) {
    Iterator<Effect> iter = this.contactEffects.iterator();
    while (iter.hasNext()) {
      ((Effect)iter.next()).applyEffect(cr);
    }
  }

  private int calcDirection(Point loc, Point target) {
    int dir, xDif2 = Point.xDif(loc, target);
    int yDif2 = Point.yDif(target, loc);
    double angle = 1.5707963267948966D;
    if (xDif2 == 0) {
      if (yDif2 != 0) {
        angle *= Math.abs(yDif2) / yDif2;
      }
    } else {
      angle = Math.atan(yDif2 / xDif2);
    }
    if (1.1780972450961724D <= angle) {
      dir = 0;
    } else if (0.39269908169872414D <= angle && angle < 1.1780972450961724D) {
      dir = 1;
    } else if (-0.39269908169872414D <= angle && angle < 0.39269908169872414D) {
      dir = 2;
    } else if (-1.1780972450961724D <= angle && angle < -0.39269908169872414D) {
      dir = 3;
    } else {
      dir = 4;
    }
    if (xDif2 < 0)
      dir = (dir + 4) % 8;
    return dir;
  }

  public Projectile spawnProjectile(Point loc, Point target) {
    Projectile proj = new Projectile(this.imgProj[calcDirection(loc, target)], loc);
    proj.addSpawnEffects(this.spawnEffects);
    proj.addContactEffects(this.contactEffects);
    proj.setTarget(target);
    return proj;
  }

  public int getDamage() {
    return ((Damage)getContactEffect(EffectType.Damage)).getDamage();
  }

  public int getAttackSpeed() {
    return this.attackSpeed;
  }

  public int getRange() {
    return this.range;
  }

  public Effect getSpawnEffect(EffectType type) {
    Iterator<Effect> iter = this.spawnEffects.iterator();
    while (iter.hasNext()) {
      Effect cur = iter.next();
      if (cur.getType() == type) {
        return cur;
      }
    }
    return null;
  }

  public Effect getContactEffect(EffectType type) {
    Iterator<Effect> iter = this.contactEffects.iterator();
    while (iter.hasNext()) {
      Effect cur = iter.next();
      if (cur.getType() == type) {
        return cur; 
      }
    }
    return null;
  }

  public void setDamage(int damage) {
    ((Damage)getContactEffect(EffectType.Damage)).setDamage(damage);
  }

  public void setAttackSpeed(int attackSpeed) {
    this.attackSpeed = attackSpeed;
  }

  public void setRange(int range) {
    this.range = range;
  }

  public void setSpawnEffect(Effect effect) {
    Iterator<Effect> iter = this.spawnEffects.iterator();
    Effect target = null;
    while (iter.hasNext() && target == null) {
      Effect cur = iter.next();
      if (cur.getType() == effect.getType()) {
        target = cur;
      }
    } 
    if (target != null) {
      this.spawnEffects.remove(target);
    }
    this.spawnEffects.add(effect);
  }

  public void setContactEffect(Effect effect) {
    Iterator<Effect> iter = this.contactEffects.iterator();
    Effect target = null;
    while (iter.hasNext() && target == null) {
      Effect cur = iter.next();
      if (cur.getType() == effect.getType()) {
        target = cur;
      }
    }
    if (target != null) {
      this.contactEffects.remove(target);
    }
    this.contactEffects.add(effect);
  }
}
