#ifndef _GRAPHICS_PIPELINE_VULKAN_H
#define _GRAPHICS_PIPELINE_VULKAN_H

#include "graphics-pipeline.hpp"

#include <vector>

#include <vulkan/vulkan.h>

// TODO: Maybe change the name of this struct so I can call the list something other than descriptorInfoList
struct DescriptorInfo {
   VkDescriptorType type;
   VkShaderStageFlags stageFlags;

   // Only one of the below properties should be set
   vector<VkDescriptorBufferInfo>* bufferDataList;
   VkDescriptorImageInfo* imageData;
};

class GraphicsPipeline_Vulkan : public GraphicsPipeline {
   public:
      GraphicsPipeline_Vulkan(VkPhysicalDevice physicalDevice, VkDevice device, VkRenderPass renderPass,
         Viewport viewport, int vertexSize);
      ~GraphicsPipeline_Vulkan();

      template<class VertexType, class IndexType>
      void bindData(const vector<VertexType>& vertices, const vector<IndexType>& indices,
         VkCommandPool commandPool, VkQueue graphicsQueue);

      void createVertexBuffer(const void* bufferData, int vertexSize, VkCommandPool commandPool,
         VkQueue graphicsQueue);
      void createIndexBuffer(const void* bufferData, int indexSize, VkCommandPool commandPool,
         VkQueue graphicsQueue);

      // Maybe I should rename these to addVertexAttribute (addVaryingAttribute) and addUniformAttribute

      void addAttribute(VkFormat format, size_t offset);

      void addDescriptorInfo(VkDescriptorType type, VkShaderStageFlags stageFlags, vector<VkDescriptorBufferInfo>* bufferData);
      void addDescriptorInfo(VkDescriptorType type, VkShaderStageFlags stageFlags, VkDescriptorImageInfo* imageData);

      void createPipeline(string vertShaderFile, string fragShaderFile);
      void createDescriptorSetLayout();
      void createDescriptorPool(vector<VkImage>& swapChainImages);
      void createDescriptorSets(vector<VkImage>& swapChainImages);

      void createRenderCommands(VkCommandBuffer& commandBuffer, uint32_t currentImage);

      void cleanup();
      void cleanupBuffers();
   
   private:
      VkShaderModule createShaderModule(const vector<char>& code);
      vector<char> readFile(const string& filename);

      VkPhysicalDevice physicalDevice;
      VkDevice device;
      VkRenderPass renderPass;

      VkPipeline pipeline;
      VkPipelineLayout pipelineLayout;

      VkVertexInputBindingDescription bindingDescription;

      vector<VkVertexInputAttributeDescription> attributeDescriptions;
      vector<DescriptorInfo> descriptorInfoList;

      VkDescriptorSetLayout descriptorSetLayout;
      VkDescriptorPool descriptorPool;
      vector<VkDescriptorSet> descriptorSets;

      size_t numVertices;
      size_t vertexCapacity;
      VkBuffer vertexBuffer;
      VkDeviceMemory vertexBufferMemory;

      size_t numIndices;
      size_t indexCapacity;
      VkBuffer indexBuffer;
      VkDeviceMemory indexBufferMemory;
};

// TODO: Probably better to template the whole class and to also combine this function
// and the constructor since I call this right after the constructor anyway
template<class VertexType, class IndexType>
void GraphicsPipeline_Vulkan::bindData(const vector<VertexType>& vertices, const vector<IndexType>& indices,
      VkCommandPool commandPool, VkQueue graphicsQueue) {
   numVertices = vertices.size();
   vertexCapacity = numVertices * 2;
   createVertexBuffer(vertices.data(), sizeof(VertexType), commandPool, graphicsQueue);

   numIndices = indices.size();
   indexCapacity = numIndices * 2;
   createIndexBuffer(indices.data(), sizeof(IndexType), commandPool, graphicsQueue);
}

#endif // _GRAPHICS_PIPELINE_VULKAN_H