#include <vulkan/vulkan.h>

#include <SDL2/SDL.h>
#include <SDL2/SDL_vulkan.h>

//#define _USE_MATH_DEFINES // Will be needed when/if I need to # include <cmath>

#define GLM_FORCE_RADIANS
#define GLM_FORCE_DEPTH_ZERO_TO_ONE
#include <glm/vec4.hpp>
#include <glm/mat4x4.hpp>

#include <iostream>
#include <vector>
#include <stdexcept>
#include <cstdlib>

#include "game-gui-sdl.hpp"

using namespace std;
using namespace glm;

const int SCREEN_WIDTH = 800;
const int SCREEN_HEIGHT = 600;

const vector<const char*> validationLayers = {
    "VK_LAYER_KHRONOS_validation"
};

#ifdef NDEBUG
   const bool enableValidationLayers = false;
#else
   const bool enableValidationLayers = true;
#endif

class VulkanGame {
   public:
      void run() {
         if (initWindow() == RTWO_ERROR) {
            return;
         }
         initVulkan();
         createInstance();
         mainLoop();
         cleanup();
      }
   private:
      GameGui_SDL gui;
      SDL_Window* window = NULL;

      VkInstance instance;

      // both SDL and GLFW create window functions return NULL on failure
      bool initWindow() {
         if (gui.Init() == RTWO_ERROR) {
            cout << "UI library could not be initialized!" << endl;
            return RTWO_ERROR;
         } else {
            // On Apple's OS X you must set the NSHighResolutionCapable Info.plist property to YES,
            // otherwise you will not receive a High DPI OpenGL canvas.

            // TODO: Move this into some generic method in game-gui-sdl
            window = SDL_CreateWindow("Vulkan Game",
               SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED,
               SCREEN_WIDTH, SCREEN_HEIGHT,
               SDL_WINDOW_VULKAN | SDL_WINDOW_SHOWN);

            if (window == NULL) {
               cout << "Window could not be created!" << endl;
               return RTWO_ERROR;
            } else {
               return RTWO_SUCCESS;
            }
         }
      }

      void initVulkan() {
         createInstance();
      }

      void createInstance() {
         VkApplicationInfo appInfo = {};
         appInfo.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
         appInfo.pApplicationName = "Vulkan Game";
         appInfo.applicationVersion = VK_MAKE_VERSION(1, 0, 0);
         appInfo.pEngineName = "No Engine";
         appInfo.engineVersion = VK_MAKE_VERSION(1, 0, 0);
         appInfo.apiVersion = VK_API_VERSION_1_0;

         VkInstanceCreateInfo createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
         createInfo.pApplicationInfo = &appInfo;

         uint32_t extensionCount;
         SDL_Vulkan_GetInstanceExtensions(window, &extensionCount, nullptr);

         vector<const char*> extensionNames(extensionCount);
         SDL_Vulkan_GetInstanceExtensions(window, &extensionCount, extensionNames.data());

         createInfo.enabledExtensionCount = extensionCount;
         createInfo.ppEnabledExtensionNames = extensionNames.data();
         createInfo.enabledLayerCount = 0;

         if (vkCreateInstance(&createInfo, nullptr, &instance) != VK_SUCCESS) {
            throw runtime_error("failed to create instance!");
         }
      }

      void mainLoop() {
         // TODO: Create some generic event-handling functions in game-gui-*
         SDL_Event e;
         bool quit = false;

         /*
         SDL_Surface* screenSurface = nullptr;
         VkSurfaceKHR surface;

         if (!SDL_Vulkan_CreateSurface(window, instance, &surface)) {
            cout << "Couild not create Vulkan surface" << endl;
         }

         screenSurface = SDL_GetWindowSurface(window);
         cout << "Got here" << endl;
         cout << (screenSurface == nullptr ? "true" : "false") << endl;

         SDL_FillRect(screenSurface, nullptr, SDL_MapRGB(screenSurface->format, 0xFF, 0xFF, 0xFF));
         cout << "Filled" << endl;

         SDL_UpdateWindowSurface(window);
         cout << "Updated" << endl;
         */

         while(!quit) {
            while (SDL_PollEvent(&e)) {
               if (e.type == SDL_QUIT) {
                  quit = true;
               }
               if (e.type == SDL_KEYDOWN) {
                  quit = true;
               }
               if (e.type == SDL_MOUSEBUTTONDOWN) {
                  quit = true;
               }
            }
         }
      }

      void cleanup() {
         vkDestroyInstance(instance, nullptr);

         // TODO: Move this into some generic method in game-gui-sdl
         SDL_DestroyWindow(window);

         gui.Shutdown();
      }
};

int main() {

#ifdef NDEBUG
   cout << "DEBUGGING IS OFF" << endl;
#else
   cout << "DEBUGGING IS ON" << endl;
#endif

   /*
   glfwInit();

   glfwWindowHint(GLFW_CLIENT_API, GLFW_NO_API);
   GLFWwindow* window = glfwCreateWindow(800, 600, "Vulkan window", nullptr, nullptr);

   uint32_t extensionCount = 0;
   vkEnumerateInstanceExtensionProperties(nullptr, &extensionCount, nullptr);

   cout << extensionCount << " extensions supported" << endl;
   */

   /*
   while(!glfwWindowShouldClose(window)) {
      glfwPollEvents();
   }

   glfwDestroyWindow(window);

   glfwTerminate();
   */

   /*
   mat4 matrix;
   vec4 vec;
   vec4 test = matrix * vec;
   */

   cout << "Starting Vulkan game..." << endl;

   VulkanGame game;

   try {
      game.run();
   } catch (const exception& e) {
      cerr << e.what() << endl;
      return EXIT_FAILURE;
   }

   cout << "Finished running the game" << endl;

   return EXIT_SUCCESS;
}
