// NEXT STEP; Modify the vertex shader

#include "logger.h"

#include <glm/mat4x4.hpp> // glm::mat4
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>

#include <GL/glew.h>
#include <GLFW/glfw3.h>

#include <cstdio>
#include <iostream>
#include <fstream>

#define _USE_MATH_DEFINES
#include <cmath>

using namespace std;
using namespace glm;

#define ONE_DEG_IN_RAD (2.0 * M_PI) / 360.0 // 0.017444444

GLuint loadShader(GLenum type, string file);

const bool FULLSCREEN = false;

void glfw_error_callback(int error, const char* description) {
   gl_log_err("GLFW ERROR: code %i msg: %s\n", error, description);
}

int main(int argc, char* argv[]) {
   cout << "New OpenGL Game" << endl;

   if (!restart_gl_log()) {}
   gl_log("starting GLFW\n%s\n", glfwGetVersionString());

   glfwSetErrorCallback(glfw_error_callback);
   if (!glfwInit()) {
      fprintf(stderr, "ERROR: could not start GLFW3\n");
      return 1;
   }

#ifdef __APPLE__
   glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
   glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
   glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
   glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
#endif

   glfwWindowHint(GLFW_SAMPLES, 4);

   GLFWwindow* window = NULL;

   int width = 640;
   int height = 480;

   if (FULLSCREEN) {
      GLFWmonitor* mon = glfwGetPrimaryMonitor();
      const GLFWvidmode* vmode = glfwGetVideoMode(mon);

      cout << "Fullscreen resolution " << vmode->width << "x" << vmode->height << endl;
      window = glfwCreateWindow(vmode->width, vmode->height, "Extended GL Init", mon, NULL);

      width = vmode->width;
      height = vmode->height;
   } else {
      window = glfwCreateWindow(width, height, "Hello Triangle", NULL, NULL);
   }

   if (!window) {
      fprintf(stderr, "ERROR: could not open window with GLFW3\n");
      glfwTerminate();
      return 1;
   }
   glfwMakeContextCurrent(window);
   glewExperimental = GL_TRUE;
   glewInit();

   // glViewport(0, 0, width*2, height*2);

   const GLubyte* renderer = glGetString(GL_RENDERER);
   const GLubyte* version = glGetString(GL_VERSION);
   printf("Renderer: %s\n", renderer);
   printf("OpenGL version supported %s\n", version);

   glEnable(GL_DEPTH_TEST);
   glDepthFunc(GL_LESS);

   glEnable(GL_CULL_FACE);
   // glCullFace(GL_BACK);
   // glFrontFace(GL_CW);

   GLfloat points[] = {
      0.0f,  0.5f,  0.0f,
     -0.5f, -0.5f,  0.0f,
      0.5f, -0.5f,  0.0f,
      0.5f, -0.5f,  0.0f,
     -0.5f, -0.5f,  0.0f,
      0.0f,  0.5f,  0.0f,
   };

   GLfloat colors[] = {
     1.0, 0.0, 0.0,
     0.0, 0.0, 1.0,
     0.0, 1.0, 0.0,
     0.0, 1.0, 0.0,
     0.0, 0.0, 1.0,
     1.0, 0.0, 0.0,
   };

   GLfloat model_mat[] = {
     1.0f, 0.0f, 0.0f, 0.0f, // column 1
     0.0f, 1.0f, 0.0f, 0.0f, // column 2
     0.0f, 0.0f, 1.0f, 0.0f, // column 3
     0.5f, 0.0f, 0.0f, 1.0f, // column 4
   };

   GLuint points_vbo = 0;
   glGenBuffers(1, &points_vbo);
   glBindBuffer(GL_ARRAY_BUFFER, points_vbo);
   glBufferData(GL_ARRAY_BUFFER, sizeof(points), points, GL_STATIC_DRAW);

   GLuint colors_vbo = 0;
   glGenBuffers(1, &colors_vbo);
   glBindBuffer(GL_ARRAY_BUFFER, colors_vbo);
   glBufferData(GL_ARRAY_BUFFER, sizeof(colors), colors, GL_STATIC_DRAW);

   GLuint vao = 0;
   glGenVertexArrays(1, &vao);
   glBindVertexArray(vao);
   glBindBuffer(GL_ARRAY_BUFFER, points_vbo);
   glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, NULL);
   glBindBuffer(GL_ARRAY_BUFFER, colors_vbo);
   glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, NULL);

   glEnableVertexAttribArray(0);
   glEnableVertexAttribArray(1);

   GLuint vs = loadShader(GL_VERTEX_SHADER, "./test.vert");
   GLuint fs = loadShader(GL_FRAGMENT_SHADER, "./test.frag");

   GLuint shader_program = glCreateProgram();
   glAttachShader(shader_program, vs);
   glAttachShader(shader_program, fs);

   glLinkProgram(shader_program);

   float speed = 1.0f;
   float last_position = 0.0f;

   float cam_speed = 1.0f;
   float cam_yaw_speed = 30.0f*ONE_DEG_IN_RAD;

   float cam_pos[] = {0.0f, 0.0f, 2.0f};
   float cam_yaw = 0.0f;

   mat4 T = translate(mat4(), vec3(-cam_pos[0], -cam_pos[1], -cam_pos[2]));
   mat4 R = rotate(mat4(), -cam_yaw, vec3(0.0f, 1.0f, 0.0f));
   mat4 view_mat = R*T;

   float near = 0.1f;
   float far = 100.0f;
   float fov = 67.0f * ONE_DEG_IN_RAD;
   float aspect = (float)width / (float)height;

   float range = tan(fov * 0.5f) * near;
   float Sx = near / (range * aspect);
   float Sy = near / range;
   float Sz = -(far + near) / (far - near);
   float Pz = -(2.0f * far * near) / (far - near);

   float proj_mat[] = {
     Sx, 0.0f, 0.0f, 0.0f,
     0.0f, Sy, 0.0f, 0.0f,
     0.0f, 0.0f, Sz, -1.0f,
     0.0f, 0.0f, Pz, 0.0f,
   };

   GLint model_mat_loc = glGetUniformLocation(shader_program, "model");
   GLint view_mat_loc = glGetUniformLocation(shader_program, "view");
   GLint proj_mat_loc = glGetUniformLocation(shader_program, "proj");

   glUseProgram(shader_program);
   glUniformMatrix4fv(model_mat_loc, 1, GL_FALSE, model_mat);
   glUniformMatrix4fv(view_mat_loc, 1, GL_FALSE, value_ptr(view_mat));
   glUniformMatrix4fv(proj_mat_loc, 1, GL_FALSE, proj_mat);

   bool cam_moved = false;

   double previous_seconds = glfwGetTime();
   while (!glfwWindowShouldClose(window)) {
      double current_seconds = glfwGetTime();
      double elapsed_seconds = current_seconds - previous_seconds;
      previous_seconds = current_seconds;

      if (fabs(last_position) > 1.0f) {
         speed = -speed;
      }

      /*
      model[12] = last_position + speed*elapsed_seconds;
      last_position = model[12];
      */

      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      glBindVertexArray(vao);

      // Each point is made of 3 floats
      int numPoints = (sizeof(points) / sizeof(float)) / 3;
      glDrawArrays(GL_TRIANGLES, 0, numPoints);

      glfwPollEvents();
      glfwSwapBuffers(window);

      if (GLFW_PRESS == glfwGetKey(window, GLFW_KEY_ESCAPE)) {
         glfwSetWindowShouldClose(window, 1);
      }

      float dist = cam_speed * elapsed_seconds;
      if (glfwGetKey(window, GLFW_KEY_A)) {
         cam_pos[0] -= cos(cam_yaw*ONE_DEG_IN_RAD)*dist;
         cam_pos[2] += sin(cam_yaw*ONE_DEG_IN_RAD)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_D)) {
         cam_pos[0] += cos(cam_yaw)*dist;
         cam_pos[2] -= sin(cam_yaw)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_W)) {
         cam_pos[0] -= sin(cam_yaw)*dist;
         cam_pos[2] -= cos(cam_yaw)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_S)) {
         cam_pos[0] += sin(cam_yaw)*dist;
         cam_pos[2] += cos(cam_yaw)*dist;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_LEFT)) {
         cam_yaw += cam_yaw_speed * elapsed_seconds;
         cam_moved = true;
      }
      if (glfwGetKey(window, GLFW_KEY_RIGHT)) {
         cam_yaw -= cam_yaw_speed * elapsed_seconds;
         cam_moved = true;
      }
      if (cam_moved) {
         T = translate(mat4(), vec3(-cam_pos[0], -cam_pos[1], -cam_pos[2]));
         R = rotate(mat4(), -cam_yaw, vec3(0.0f, 1.0f, 0.0f));
         view_mat = R*T;

         glUniformMatrix4fv(view_mat_loc, 1, GL_FALSE, value_ptr(view_mat));
         cam_moved = false;
      }
   }

   glfwTerminate();
   return 0;
}

GLuint loadShader(GLenum type, string file) {
  cout << "Loading shader from file " << file << endl;

  ifstream shaderFile(file);
  GLuint shaderId = 0;

  if (shaderFile.is_open()) {
    string line, shaderString;

    while(getline(shaderFile, line)) {
      shaderString += line + "\n";
    }
    shaderFile.close();
    const char* shaderCString = shaderString.c_str();

    shaderId = glCreateShader(type);
    glShaderSource(shaderId, 1, &shaderCString, NULL);
    glCompileShader(shaderId);

    cout << "Loaded successfully" << endl;
  } else {
    cout << "Failed to loade the file" << endl;
  }

  return shaderId;
}
