#include <vulkan/vulkan.h>

#include <SDL2/SDL.h>
#include <SDL2/SDL_vulkan.h>

//#define _USE_MATH_DEFINES // Will be needed when/if I need to # include <cmath>

#define GLM_FORCE_RADIANS
#define GLM_FORCE_DEPTH_ZERO_TO_ONE
#include <glm/vec4.hpp>
#include <glm/mat4x4.hpp>

#include <iostream>
#include <vector>
#include <stdexcept>
#include <cstdlib>

#include "game-gui-sdl.hpp"

using namespace std;
using namespace glm;

bool checkValidationLayerSupport();
vector<const char*> getRequiredExtensions(SDL_Window* window);

const int SCREEN_WIDTH = 800;
const int SCREEN_HEIGHT = 600;

const vector<const char*> validationLayers = {
    "VK_LAYER_KHRONOS_validation"
};

#ifdef NDEBUG
   const bool enableValidationLayers = false;
#else
   const bool enableValidationLayers = true;
#endif

static VKAPI_ATTR VkBool32 VKAPI_CALL debugCallback(
      VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity,
      VkDebugUtilsMessageTypeFlagsEXT messageType,
      const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData,
      void* pUserData) {
   cerr << "validation layer: " << pCallbackData->pMessage << endl;

   return VK_FALSE;
}

VkResult CreateDebugUtilsMessengerEXT(VkInstance instance,
      const VkDebugUtilsMessengerCreateInfoEXT* pCreateInfo,
      const VkAllocationCallbacks* pAllocator,
      VkDebugUtilsMessengerEXT* pDebugMessenger) {
   auto func = (PFN_vkCreateDebugUtilsMessengerEXT) vkGetInstanceProcAddr(
      instance, "vkCreateDebugUtilsMessengerEXT");

   if (func != nullptr) {
      return func(instance, pCreateInfo, pAllocator, pDebugMessenger);
   } else {
      return VK_ERROR_EXTENSION_NOT_PRESENT;
   }
}

class VulkanGame {
   public:
      void run() {
         if (initWindow() == RTWO_ERROR) {
            return;
         }
         initVulkan();
         createInstance();
         mainLoop();
         cleanup();
      }
   private:
      GameGui_SDL gui;
      SDL_Window* window = NULL;

      VkInstance instance;
      VkDebugUtilsMessengerEXT debugMessenger;

      // both SDL and GLFW create window functions return NULL on failure
      bool initWindow() {
         if (gui.Init() == RTWO_ERROR) {
            cout << "UI library could not be initialized!" << endl;
            return RTWO_ERROR;
         } else {
            // On Apple's OS X you must set the NSHighResolutionCapable Info.plist property to YES,
            // otherwise you will not receive a High DPI OpenGL canvas.

            // TODO: Move this into some generic method in game-gui-sdl
            window = SDL_CreateWindow("Vulkan Game",
               SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED,
               SCREEN_WIDTH, SCREEN_HEIGHT,
               SDL_WINDOW_VULKAN | SDL_WINDOW_SHOWN);

            if (window == NULL) {
               cout << "Window could not be created!" << endl;
               return RTWO_ERROR;
            } else {
               return RTWO_SUCCESS;
            }
         }
      }

      void initVulkan() {
         createInstance();
         setupDebugMessenger();
      }

      void createInstance() {
         if (enableValidationLayers && !checkValidationLayerSupport()) {
            throw runtime_error("validation layers requested, but not available!");
         }

         VkApplicationInfo appInfo = {};
         appInfo.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
         appInfo.pApplicationName = "Vulkan Game";
         appInfo.applicationVersion = VK_MAKE_VERSION(1, 0, 0);
         appInfo.pEngineName = "No Engine";
         appInfo.engineVersion = VK_MAKE_VERSION(1, 0, 0);
         appInfo.apiVersion = VK_API_VERSION_1_0;

         VkInstanceCreateInfo createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
         createInfo.pApplicationInfo = &appInfo;

         auto extensions = getRequiredExtensions(window);
         createInfo.enabledExtensionCount = static_cast<uint32_t>(extensions.size());
         createInfo.ppEnabledExtensionNames = extensions.data();

         if (enableValidationLayers) {
            createInfo.enabledLayerCount = static_cast<uint32_t>(validationLayers.size());
            createInfo.ppEnabledLayerNames = validationLayers.data();
         } else {
            createInfo.enabledLayerCount = 0;
         }

         if (vkCreateInstance(&createInfo, nullptr, &instance) != VK_SUCCESS) {
            throw runtime_error("failed to create instance!");
         }
      }

      void setupDebugMessenger() {
         if (!enableValidationLayers) return;

         VkDebugUtilsMessengerCreateInfoEXT createInfo = {};
         createInfo.sType = VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT;
         createInfo.messageSeverity = VK_DEBUG_UTILS_MESSAGE_SEVERITY_VERBOSE_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_SEVERITY_WARNING_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_SEVERITY_ERROR_BIT_EXT;
         createInfo.messageType = VK_DEBUG_UTILS_MESSAGE_TYPE_GENERAL_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_TYPE_VALIDATION_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_TYPE_PERFORMANCE_BIT_EXT;
         createInfo.pfnUserCallback = debugCallback;
         createInfo.pUserData = nullptr;

         if (CreateDebugUtilsMessengerEXT(instance, &createInfo, nullptr, &debugMessenger) != VK_SUCCESS) {
            throw runtime_error("failed to setup debug messenger!");
         }
      }

      void mainLoop() {
         // TODO: Create some generic event-handling functions in game-gui-*
         SDL_Event e;
         bool quit = false;

         /*
         SDL_Surface* screenSurface = nullptr;
         VkSurfaceKHR surface;

         if (!SDL_Vulkan_CreateSurface(window, instance, &surface)) {
            cout << "Couild not create Vulkan surface" << endl;
         }

         screenSurface = SDL_GetWindowSurface(window);
         cout << "Got here" << endl;
         cout << (screenSurface == nullptr ? "true" : "false") << endl;

         SDL_FillRect(screenSurface, nullptr, SDL_MapRGB(screenSurface->format, 0xFF, 0xFF, 0xFF));
         cout << "Filled" << endl;

         SDL_UpdateWindowSurface(window);
         cout << "Updated" << endl;
         */

         while (!quit) {
            while (SDL_PollEvent(&e)) {
               if (e.type == SDL_QUIT) {
                  quit = true;
               }
               if (e.type == SDL_KEYDOWN) {
                  quit = true;
               }
               if (e.type == SDL_MOUSEBUTTONDOWN) {
                  quit = true;
               }
            }
         }
      }

      void cleanup() {
         vkDestroyInstance(instance, nullptr);

         // TODO: Move this into some generic method in game-gui-sdl
         SDL_DestroyWindow(window);

         gui.Shutdown();
      }
};

vector<const char*> getRequiredExtensions(SDL_Window* window) {
   uint32_t extensionCount;
   SDL_Vulkan_GetInstanceExtensions(window, &extensionCount, nullptr);

   vector<const char*> extensions(extensionCount);
   SDL_Vulkan_GetInstanceExtensions(window, &extensionCount, extensions.data());

   if (enableValidationLayers) {
      extensions.push_back(VK_EXT_DEBUG_UTILS_EXTENSION_NAME);
   }

   return extensions;
}

bool checkValidationLayerSupport() {
   uint32_t layerCount;
   vkEnumerateInstanceLayerProperties(&layerCount, nullptr);

   vector<VkLayerProperties> availableLayers(layerCount);
   vkEnumerateInstanceLayerProperties(&layerCount, availableLayers.data());

   for (const char* layerName : validationLayers) {
      bool layerFound = false;

      for (const auto& layerProperties : availableLayers) {
         if (strcmp(layerName, layerProperties.layerName) == 0) {
            layerFound = true;
            break;
         }
      }

      if (!layerFound) {
         return false;
      }
   }

   return true;
}

int main() {

#ifdef NDEBUG
   cout << "DEBUGGING IS OFF" << endl;
#else
   cout << "DEBUGGING IS ON" << endl;
#endif
   /*
   mat4 matrix;
   vec4 vec;
   vec4 test = matrix * vec;
   */

   cout << "Starting Vulkan game..." << endl;

   VulkanGame game;

   try {
      game.run();
   } catch (const exception& e) {
      cerr << e.what() << endl;
      return EXIT_FAILURE;
   }

   cout << "Finished running the game" << endl;

   return EXIT_SUCCESS;
}
