#ifndef _VULKAN_BUFFER_H
#define _VULKAN_BUFFER_H

/* 
* This class is intended to be used with Storage Buffers and Uniform Buffers.
*/

template<class T>
class VulkanBuffer {

   public:

      // TODO: Make these private (maybe make a getter for numObjects)
      // Externally, they are only used in resizeBufferSet
      size_t capacity;

      // temp field to help with ubo+ssbo resizing until they are added to this class
      // See if I need a separate field for this or if I can use other fields to check for this
      // Maybe compare uniform or storage buffer size to the size of the memory allocated here
      bool resized;

      VulkanBuffer();
      VulkanBuffer(size_t capacity, size_t range, size_t minOffsetAlignment);

      VulkanBuffer(const VulkanBuffer<T>&) = delete;
      VulkanBuffer(VulkanBuffer<T>&& other);

      ~VulkanBuffer();

      VulkanBuffer<T>& operator=(const VulkanBuffer<T>&) = delete;
      VulkanBuffer<T>& operator=(VulkanBuffer<T>&& other) noexcept;

      void resize();

      T& get(uint32_t index);
      void add(T obj);

      size_t memorySize();

   private:

      size_t alignment;
      size_t range;
      //size_t capacity;
      size_t numObjects;

      T* rawData;
      vector<void*> mappedData;

      size_t memRequirement(size_t capacity);
      size_t memOffset(uint32_t index);
};

// Currently, for SSBOs, I store the per-object values (usually just the model matrix), on each object, so they
// are not in their own array and therefore cannot be pushed to the GPU as one block. The updates must happen
// separately per object.

// Since Sascha WIllems' dynamic UBO example works the same way (iirc), I can implement dynamic UBOs like that as well
// for now. Would be nice to plan for potentially storing the ubo data on the CPU in a contiguous block in the future,
// assuming that would make updates easier. Keep in mind that this only makes sense if all or most of the objects
// in the ubo get updated every frame.

// ============================= TODO: Also, check when it makes sense to have a staging buffer for copying data to the GPU
// and see if I actually need to use it everywhere I currently am. I think this is mentioned in Sascha WIllems dubo example
// or some other Vulkan website I recently bookmarked

template<class T>
VulkanBuffer<T>::VulkanBuffer()
                              : alignment(0)
                              , range(0)
                              , capacity(0)
                              , numObjects(0)
                              , resized(false)
                              , rawData(nullptr)
                              , mappedData() {
}

template<class T>
VulkanBuffer<T>::VulkanBuffer(size_t capacity, size_t range, size_t minOffsetAlignment)
                              : alignment(range)
                              , range(range / sizeof(T))
                              , capacity(capacity)
                              , numObjects(0)
                              , resized(false)
                              , rawData(nullptr)
                              , mappedData() {
   if (minOffsetAlignment > 0) {
      alignment = (alignment + minOffsetAlignment - 1) & ~(minOffsetAlignment - 1);
   }

   rawData = (T*)malloc(memRequirement(capacity));
}

template<class T>
VulkanBuffer<T>::VulkanBuffer(VulkanBuffer<T>&& other) {
   // TODO: Implement
}

template<class T>
VulkanBuffer<T>::~VulkanBuffer() {
   if (rawData != nullptr) {
      free(rawData);
   }
}

template<class T>
VulkanBuffer<T>& VulkanBuffer<T>::operator=(VulkanBuffer<T>&& other) noexcept {
   if (this != &other) {
      capacity = other.capacity;
      numObjects = other.numObjects;
      resized = other.resized;

      alignment = other.alignment;
      range = other.range;

      if (rawData != nullptr) {
         free(rawData);
      }

      rawData = other.rawData;

      other.capacity = 0;
      other.numObjects = 0;
      other.range = 0;

      other.rawData = nullptr;
   }

   return *this;
}

template<class T>
void VulkanBuffer<T>::resize() {
   resized = false;
}

template<class T>
T& VulkanBuffer<T>::get(uint32_t index) {
   // TODO: Check that index < numObjects

   T* obj = (T*)((size_t)rawData + memOffset(index));
   return *obj;
}

template<class T>
void VulkanBuffer<T>::add(T obj) {
   if (numObjects == capacity) {
      // Once I add Vulkan buffer objects in here, make sure this doesn't overlap with resizeBufferSet
      resized = true;

      size_t oldMemReq = memRequirement(capacity);

      capacity *= 2;

      size_t newMemReq = memRequirement(capacity);

      T* newData = (T*)malloc(newMemReq);
      // TODO: Check for failure

      memcpy(newData, rawData, oldMemReq);

      free(rawData);
      rawData = newData;
   }

   T* ptr = (T*)((size_t)rawData + memOffset(numObjects));
   *ptr = obj;

   numObjects++;
}

template<class T>
size_t VulkanBuffer<T>::memorySize() {
   return memRequirement(capacity);
}

template<class T>
size_t VulkanBuffer<T>::memRequirement(size_t capacity) {
   return (capacity / range) * alignment + (capacity % range) * sizeof(T);
}

template<class T>
size_t VulkanBuffer<T>::memOffset(uint32_t index) {
   return (index / range) * alignment + (index % range) * sizeof(T);
}

#endif // _VULKAN_BUFFER_H
