#ifndef _VULKAN_UTILS_H
#define _VULKAN_UTILS_H

#include <optional>
#include <string>
#include <vector>

#include <vulkan/vulkan.h>

#include <SDL2/SDL.h>
#include <SDL2/SDL_image.h>
#include <SDL2/SDL_vulkan.h>

using namespace std;

struct QueueFamilyIndices {
   optional<uint32_t> graphicsFamily;
   optional<uint32_t> presentFamily;

   bool isComplete() {
      return graphicsFamily.has_value() && presentFamily.has_value();
   }
};

struct SwapChainSupportDetails {
   VkSurfaceCapabilitiesKHR capabilities;
   vector<VkSurfaceFormatKHR> formats;
   vector<VkPresentModeKHR> presentModes;
};

struct VulkanImage {
   VkImage image;
   VkDeviceMemory imageMemory;
   VkImageView imageView;
};

class VulkanUtils {
   public:
      static bool checkValidationLayerSupport(const vector<const char*> &validationLayers);

      static VkResult createDebugUtilsMessengerEXT(VkInstance instance,
            const VkDebugUtilsMessengerCreateInfoEXT* pCreateInfo,
            const VkAllocationCallbacks* pAllocator,
            VkDebugUtilsMessengerEXT* pDebugMessenger);

      static void destroyDebugUtilsMessengerEXT(VkInstance instance,
            VkDebugUtilsMessengerEXT debugMessenger,
            const VkAllocationCallbacks* pAllocator);

      static QueueFamilyIndices findQueueFamilies(VkPhysicalDevice physicalDevice, VkSurfaceKHR surface);
      static bool checkDeviceExtensionSupport(VkPhysicalDevice physicalDevice,
            const vector<const char*>& deviceExtensions);
      static SwapChainSupportDetails querySwapChainSupport(VkPhysicalDevice physicalDevice,
            VkSurfaceKHR surface);
      static VkSurfaceFormatKHR chooseSwapSurfaceFormat(const vector<VkSurfaceFormatKHR>& availableFormats);
      static VkPresentModeKHR chooseSwapPresentMode(const vector<VkPresentModeKHR>& availablePresentModes);
      static VkExtent2D chooseSwapExtent(const VkSurfaceCapabilitiesKHR& capabilities, int width, int height);
      static VkImageView createImageView(VkDevice device, VkImage image, VkFormat format,
            VkImageAspectFlags aspectFlags);
      static VkFormat findSupportedFormat(VkPhysicalDevice physicalDevice, const vector<VkFormat>& candidates,
            VkImageTiling tiling, VkFormatFeatureFlags features);
      static void createBuffer(VkDevice device, VkPhysicalDevice physicalDevice, VkDeviceSize size,
            VkBufferUsageFlags usage, VkMemoryPropertyFlags properties, VkBuffer& buffer,
            VkDeviceMemory& bufferMemory);
      static uint32_t findMemoryType(VkPhysicalDevice physicalDevice, uint32_t typeFilter,
            VkMemoryPropertyFlags properties);

      static void createVulkanImageFromFile(VkDevice device, VkPhysicalDevice physicalDevice,
            VkCommandPool commandPool, string filename, VulkanImage& image, VkQueue graphicsQueue);
      static void createVulkanImageFromSDLTexture(VkDevice device, VkPhysicalDevice physicalDevice,
            SDL_Texture* texture, VulkanImage& image);
      static void createDepthImage(VkDevice device, VkPhysicalDevice physicalDevice, VkCommandPool commandPool,
            VkFormat depthFormat, VkExtent2D extent, VulkanImage& image, VkQueue graphicsQueue);
      static void createImage(VkDevice device, VkPhysicalDevice physicalDevice, uint32_t width, uint32_t height,
            VkFormat format, VkImageTiling tiling, VkImageUsageFlags usage, VkMemoryPropertyFlags properties,
            VulkanImage& image);

      static void transitionImageLayout(VkDevice device, VkCommandPool commandPool, VkImage image,
            VkFormat format, VkImageLayout oldLayout, VkImageLayout newLayout, VkQueue graphicsQueue);
      static VkCommandBuffer beginSingleTimeCommands(VkDevice device, VkCommandPool commandPool);
      static void endSingleTimeCommands(VkDevice device, VkCommandPool commandPool,
            VkCommandBuffer commandBuffer, VkQueue graphicsQueue);
      static void copyBufferToImage(VkDevice device, VkCommandPool commandPool, VkBuffer buffer, VkImage image,
            uint32_t width, uint32_t height, VkQueue graphicsQueue);

      static bool hasStencilComponent(VkFormat format);

      static void destroyVulkanImage(VkDevice& device, VulkanImage& image);
};

#endif // _VULKAN_UTILS_H