#ifndef _VULKAN_GAME_H
#define _VULKAN_GAME_H

#include <algorithm>
#include <chrono>
#include <map>

#define GLM_FORCE_RADIANS
#define GLM_FORCE_DEPTH_ZERO_TO_ONE // Since, in Vulkan, the depth range is 0 to 1 instead of -1 to 1
#define GLM_FORCE_RIGHT_HANDED

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>

#include <vulkan/vulkan.h>

#include <SDL2/SDL.h>
#include <SDL2/SDL_ttf.h>

#include "consts.hpp"
#include "vulkan-utils.hpp"
#include "graphics-pipeline_vulkan.hpp"

#include "game-gui-sdl.hpp"

#include "gui/screen.hpp"

using namespace glm;
using namespace std::chrono;

#ifdef NDEBUG
   const bool ENABLE_VALIDATION_LAYERS = false;
#else
   const bool ENABLE_VALIDATION_LAYERS = true;
#endif

struct OverlayVertex {
   vec3 pos;
   vec2 texCoord;
};

struct ModelVertex {
   vec3 pos;
   vec3 color;
   vec2 texCoord;
   unsigned int objIndex;
};

struct ShipVertex {
   vec3 pos;
   vec3 color;
   vec3 normal;
   unsigned int objIndex;
};

struct AsteroidVertex {
   vec3 pos;
   vec3 color;
   vec3 normal;
   unsigned int objIndex;
};

struct LaserVertex {
   vec3 pos;
   vec2 texCoord;
   unsigned int objIndex;
};

struct ExplosionVertex {
   vec3 particleStartVelocity;
   float particleStartTime;
   unsigned int objIndex;
};

struct SSBO_ModelObject {
   alignas(16) mat4 model;
};

struct SSBO_Asteroid {
   alignas(16) mat4 model;
   alignas(4) float hp;
   alignas(4) unsigned int deleted;
};

struct SSBO_Laser {
   alignas(16) mat4 model;
   alignas(4) vec3 color;
   alignas(4) unsigned int deleted;
};

struct SSBO_Explosion {
   alignas(16) mat4 model;
   alignas(4) float explosionStartTime;
   alignas(4) float explosionDuration;
   alignas(4) unsigned int deleted;
};

struct UBO_VP_mats {
   alignas(16) mat4 view;
   alignas(16) mat4 proj;
};

struct UBO_Explosion {
   alignas(16) mat4 view;
   alignas(16) mat4 proj;
   alignas(4) float cur_time;
};

// TODO: Change the index type to uint32_t and check the Vulkan Tutorial loading model section as a reference
// TODO: Create a typedef for index type so I can easily change uin16_t to something else later
// TODO: Maybe create a typedef for each of the templated SceneObject types
template<class VertexType, class SSBOType>
struct SceneObject {
   vector<VertexType> vertices;
   vector<uint16_t> indices;
   SSBOType ssbo;

   mat4 model_base;
   mat4 model_transform;

   bool modified;

   // TODO: Figure out if I should make child classes that have these fields instead of putting them in the
   // parent class
   vec3 center; // currently only matters for asteroids
   float radius; // currently only matters for asteroids
   SceneObject<AsteroidVertex, SSBO_Asteroid>* targetAsteroid; // currently only used for lasers
};

// TODO: Have to figure out how to include an optional ssbo parameter for each object
// Could probably use the same approach to make indices optional
// Figure out if there are sufficient use cases to make either of these optional or is it fine to make
// them mamdatory

// TODO: Make a singleton timer class instead
static float curTime;


// TODO: Look into using dynamic_cast to check types of SceneObject and EffectOverTime

struct BaseEffectOverTime {
   bool deleted;

   virtual void applyEffect() = 0;

   BaseEffectOverTime() :
         deleted(false) {
   }

   virtual ~BaseEffectOverTime() {
   }
};

template<class VertexType, class SSBOType>
struct EffectOverTime : public BaseEffectOverTime {
   GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline;
   vector<SceneObject<VertexType, SSBOType>>& objects;
   unsigned int objectIndex;
   size_t effectedFieldOffset;
   float startValue;
   float startTime;
   float changePerSecond;

   EffectOverTime(GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline,
         vector<SceneObject<VertexType, SSBOType>>& objects, unsigned int objectIndex,
         size_t effectedFieldOffset, float changePerSecond) :
         pipeline(pipeline),
         objects(objects),
         objectIndex(objectIndex),
         effectedFieldOffset(effectedFieldOffset),
         startTime(curTime),
         changePerSecond(changePerSecond) {
      size_t ssboOffset = offset_of(&SceneObject<VertexType, SSBOType>::ssbo);

      unsigned char* effectedFieldPtr = reinterpret_cast<unsigned char*>(&objects[objectIndex]) +
         ssboOffset + effectedFieldOffset;

      startValue = *reinterpret_cast<float*>(effectedFieldPtr);
   }

   void applyEffect() {
      if (objects[objectIndex].ssbo.deleted) {
         this->deleted = true;
         return;
      }

      size_t ssboOffset = offset_of(&SceneObject<VertexType, SSBOType>::ssbo);

      unsigned char* effectedFieldPtr = reinterpret_cast<unsigned char*>(&objects[objectIndex]) +
         ssboOffset + effectedFieldOffset;

      *reinterpret_cast<float*>(effectedFieldPtr) = startValue + (curTime - startTime) * changePerSecond;

      objects[objectIndex].modified = true;
   }
};

class VulkanGame {
   public:
      VulkanGame();
      ~VulkanGame();

      void run(int width, int height, unsigned char guiFlags);

      void goToScreen(Screen* screen);
      void quitGame();

      map<ScreenType, Screen*> screens;
      Screen* currentScreen;

      TTF_Font* lazyFont;
      TTF_Font* proggyFont;

      int score;
      float fps;

      GraphicsPipeline_Vulkan<OverlayVertex, void*> overlayPipeline;

      GraphicsPipeline_Vulkan<ModelVertex, SSBO_ModelObject> modelPipeline;

      GraphicsPipeline_Vulkan<ShipVertex, SSBO_ModelObject> shipPipeline;

      GraphicsPipeline_Vulkan<AsteroidVertex, SSBO_Asteroid> asteroidPipeline;

      GraphicsPipeline_Vulkan<LaserVertex, SSBO_Laser> laserPipeline;

      GraphicsPipeline_Vulkan<ExplosionVertex, SSBO_Explosion> explosionPipeline;

   private:
      static VKAPI_ATTR VkBool32 VKAPI_CALL debugCallback(
         VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity,
         VkDebugUtilsMessageTypeFlagsEXT messageType,
         const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData,
         void* pUserData);

      const float NEAR_CLIP = 0.1f;
      const float FAR_CLIP = 100.0f;
      const float FOV_ANGLE = 67.0f; // means the camera lens goes from -33 deg to 33 def

      const int EXPLOSION_PARTICLE_COUNT = 300;
      const vec3 LASER_COLOR = vec3(0.2f, 1.0f, 0.2f);

      bool done;

      vec3 cam_pos;

      // TODO: Good place to start using smart pointers
      GameGui* gui;

      SDL_version sdlVersion;
      SDL_Window* window = nullptr;
      SDL_Renderer* renderer = nullptr;

      SDL_Texture* uiOverlay = nullptr;

      VkInstance instance;
      VkDebugUtilsMessengerEXT debugMessenger;
      VkSurfaceKHR surface; // TODO: Change the variable name to vulkanSurface
      VkPhysicalDevice physicalDevice = VK_NULL_HANDLE;
      VkDevice device;

      VkQueue graphicsQueue;
      VkQueue presentQueue;

      // TODO: Maybe make a swapchain struct for convenience
      VkSurfaceFormatKHR swapChainSurfaceFormat;
      VkPresentModeKHR swapChainPresentMode;
      VkExtent2D swapChainExtent;
      uint32_t swapChainMinImageCount;
      uint32_t swapChainImageCount;
      VkSwapchainKHR swapChain;
      vector<VkImage> swapChainImages;
      vector<VkImageView> swapChainImageViews;
      vector<VkFramebuffer> swapChainFramebuffers;

      VkRenderPass renderPass;

      VkCommandPool resourceCommandPool;

      vector<VkCommandPool> commandPools;
      vector<VkCommandBuffer> commandBuffers;

      VulkanImage depthImage;

      // These are per frame
      vector<VkSemaphore> imageAcquiredSemaphores;
      vector<VkSemaphore> renderCompleteSemaphores;

      // These are per swap chain image
      vector<VkFence> inFlightFences;

      uint32_t imageIndex;
      uint32_t currentFrame;

      bool framebufferResized;

      VkDescriptorPool imguiDescriptorPool;

      VkSampler textureSampler;

      VulkanImage sdlOverlayImage;
      VkDescriptorImageInfo sdlOverlayImageDescriptor;

      VulkanImage floorTextureImage;
      VkDescriptorImageInfo floorTextureImageDescriptor;

      VulkanImage laserTextureImage;
      VkDescriptorImageInfo laserTextureImageDescriptor;

      mat4 viewMat, projMat;

      // TODO: Maybe make the ubo objects part of the pipeline class since there's only one ubo
      // per pipeline.
      // Or maybe create a higher level wrapper around GraphicsPipeline_Vulkan to hold things like
      // the objects vector, the ubo, and the ssbo

      // TODO: Rename *_VP_mats to *_uniforms and possibly use different types for each one
      // if there is a need to add other uniform variables to one or more of the shaders

      vector<SceneObject<OverlayVertex, void*>> overlayObjects;

      vector<SceneObject<ModelVertex, SSBO_ModelObject>> modelObjects;

      vector<VkBuffer> uniformBuffers_modelPipeline;
      vector<VkDeviceMemory> uniformBuffersMemory_modelPipeline;
      vector<VkDescriptorBufferInfo> uniformBufferInfoList_modelPipeline;

      UBO_VP_mats object_VP_mats;

      vector<SceneObject<ShipVertex, SSBO_ModelObject>> shipObjects;

      vector<VkBuffer> uniformBuffers_shipPipeline;
      vector<VkDeviceMemory> uniformBuffersMemory_shipPipeline;
      vector<VkDescriptorBufferInfo> uniformBufferInfoList_shipPipeline;

      UBO_VP_mats ship_VP_mats;

      vector<SceneObject<AsteroidVertex, SSBO_Asteroid>> asteroidObjects;

      vector<VkBuffer> uniformBuffers_asteroidPipeline;
      vector<VkDeviceMemory> uniformBuffersMemory_asteroidPipeline;
      vector<VkDescriptorBufferInfo> uniformBufferInfoList_asteroidPipeline;

      UBO_VP_mats asteroid_VP_mats;

      vector<SceneObject<LaserVertex, SSBO_Laser>> laserObjects;

      vector<VkBuffer> uniformBuffers_laserPipeline;
      vector<VkDeviceMemory> uniformBuffersMemory_laserPipeline;
      vector<VkDescriptorBufferInfo> uniformBufferInfoList_laserPipeline;

      UBO_VP_mats laser_VP_mats;

      vector<SceneObject<ExplosionVertex, SSBO_Explosion>> explosionObjects;

      vector<VkBuffer> uniformBuffers_explosionPipeline;
      vector<VkDeviceMemory> uniformBuffersMemory_explosionPipeline;
      vector<VkDescriptorBufferInfo> uniformBufferInfoList_explosionPipeline;

      UBO_Explosion explosion_UBO;

      vector<BaseEffectOverTime*> effects;

      time_point<steady_clock> startTime;
      float prevTime, elapsedTime;

      float fpsStartTime;
      int frameCount;

      float shipSpeed = 0.5f;
      float asteroidSpeed = 2.0f;

      float spawnRate_asteroid = 0.5;
      float lastSpawn_asteroid;

      unsigned int leftLaserIdx = -1;
      EffectOverTime<AsteroidVertex, SSBO_Asteroid>* leftLaserEffect = nullptr;

      unsigned int rightLaserIdx = -1;
      EffectOverTime<AsteroidVertex, SSBO_Asteroid>* rightLaserEffect = nullptr;

      bool initUI(int width, int height, unsigned char guiFlags);
      void initVulkan();
      void initGraphicsPipelines();
      void initMatrices();
      void mainLoop();
      void updateScene();
      void cleanup();

      void createVulkanInstance(const vector<const char*>& validationLayers);
      void setupDebugMessenger();
      void populateDebugMessengerCreateInfo(VkDebugUtilsMessengerCreateInfoEXT& createInfo);
      void createVulkanSurface();
      void pickPhysicalDevice(const vector<const char*>& deviceExtensions);
      bool isDeviceSuitable(VkPhysicalDevice physicalDevice, const vector<const char*>& deviceExtensions);
      void createLogicalDevice(const vector<const char*>& validationLayers,
         const vector<const char*>& deviceExtensions);
      void chooseSwapChainProperties();
      void createSwapChain();
      void createImageViews();
      void createRenderPass();
      VkFormat findDepthFormat(); // TODO: Declare/define (in the cpp file) this function in some util functions section
      void createResourceCommandPool();
      void createCommandPools();
      void createImageResources();
      void createFramebuffers();
      void createCommandBuffers();
      void createSyncObjects();

      void createTextureSampler();

      void createImguiDescriptorPool();
      void destroyImguiDescriptorPool();

      // TODO: Since addObject() returns a reference to the new object now,
      // stop using objects.back() to access the object that was just created
      template<class VertexType, class SSBOType>
      SceneObject<VertexType, SSBOType>& addObject(
            vector<SceneObject<VertexType, SSBOType>>& objects,
            GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline,
            const vector<VertexType>& vertices, vector<uint16_t> indices, SSBOType ssbo,
            bool pipelinesCreated);

      template<class VertexType, class SSBOType>
      void updateObject(vector<SceneObject<VertexType, SSBOType>>& objects,
            GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline, size_t index);

      template<class VertexType, class SSBOType>
      void updateObjectVertices(GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline,
            SceneObject<VertexType, SSBOType>& obj, size_t index);

      template<class VertexType>
      vector<VertexType> addVertexNormals(vector<VertexType> vertices);

      template<class VertexType>
      vector<VertexType> addObjectIndex(unsigned int objIndex, vector<VertexType> vertices);

      template<class VertexType, class SSBOType>
      void centerObject(SceneObject<VertexType, SSBOType>& object);

      void addLaser(vec3 start, vec3 end, vec3 color, float width);
      void translateLaser(size_t index, const vec3& translation);
      void updateLaserTarget(size_t index);
      bool getLaserAndAsteroidIntersection(SceneObject<AsteroidVertex, SSBO_Asteroid>& asteroid,
            vec3& start, vec3& end, vec3& intersection);

      void addExplosion(mat4 model_mat, float duration, float cur_time);

      void createBufferSet(VkDeviceSize bufferSize, VkBufferUsageFlags flags,
            vector<VkBuffer>& buffers, vector<VkDeviceMemory>& buffersMemory,
            vector<VkDescriptorBufferInfo>& bufferInfoList);

      void renderFrame();
      void presentFrame();

      void recreateSwapChain();

      void cleanupSwapChain();
};

// Start of specialized no-op functions

template<>
inline void VulkanGame::centerObject(SceneObject<ExplosionVertex, SSBO_Explosion>& object) {
}

// End of specialized no-op functions

// TODO: Right now, it's basically necessary to pass the identity matrix in for ssbo.model
// and to change the model matrix later by setting model_transform and then calling updateObject()
// Figure out a better way to allow the model matrix to be set during objecting creation

// TODO: Maybe return a reference to the object from this method if I decide that updating it
// immediately after creation is a good idea (such as setting model_base)
// Currently, model_base is set like this in a few places and the radius is set for asteroids
// to account for scaling
template<class VertexType, class SSBOType>
SceneObject<VertexType, SSBOType>& VulkanGame::addObject(
      vector<SceneObject<VertexType, SSBOType>>& objects,
      GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline,
      const vector<VertexType>& vertices, vector<uint16_t> indices, SSBOType ssbo,
      bool pipelinesCreated) {
   // TODO: Use the model field of ssbo to set the object's model_base
   // currently, the passed in model is useless since it gets overridden in updateObject() anyway
   size_t numVertices = pipeline.getNumVertices();

   for (uint16_t& idx : indices) {
      idx += numVertices;
   }

   objects.push_back({ vertices, indices, ssbo, mat4(1.0f), mat4(1.0f), false });

   SceneObject<VertexType, SSBOType>& obj = objects.back();

   if (!is_same_v<VertexType, LaserVertex> && !is_same_v<VertexType, ExplosionVertex>) {
      centerObject(obj);
   }

   bool storageBufferResized = pipeline.addObject(obj.vertices, obj.indices, obj.ssbo,
      resourceCommandPool, graphicsQueue);

   if (pipelinesCreated) {
      vkDeviceWaitIdle(device);

      for (uint32_t i = 0; i < swapChainImageCount; i++) {
         vkFreeCommandBuffers(device, commandPools[i], 1, &commandBuffers[i]);
      }

      // TODO: The pipeline recreation only has to be done once per frame where at least
      // one SSBO is resized.
      // Refactor the logic to check for any resized SSBOs after all objects for the frame
      // are created and then recreate each of the corresponding pipelines only once per frame
      if (storageBufferResized) {
         pipeline.createPipeline(pipeline.vertShaderFile, pipeline.fragShaderFile);
         pipeline.createDescriptorPool(swapChainImages);
         pipeline.createDescriptorSets(swapChainImages);
      }

      createCommandBuffers();
   }

   return obj;
}

// TODO: Just pass in the single object instead of a list of all of them
template<class VertexType, class SSBOType>
void VulkanGame::updateObject(vector<SceneObject<VertexType, SSBOType>>& objects,
      GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline, size_t index) {
   SceneObject<VertexType, SSBOType>& obj = objects[index];

   obj.ssbo.model = obj.model_transform * obj.model_base;
   obj.center = vec3(obj.ssbo.model * vec4(0.0f, 0.0f, 0.0f, 1.0f));

   pipeline.updateObject(index, obj.ssbo);

   obj.modified = false;
}

template<class VertexType, class SSBOType>
void VulkanGame::updateObjectVertices(GraphicsPipeline_Vulkan<VertexType, SSBOType>& pipeline,
      SceneObject<VertexType, SSBOType>& obj, size_t index) {
   pipeline.updateObjectVertices(index, obj.vertices, resourceCommandPool, graphicsQueue);
}

template<class VertexType>
vector<VertexType> VulkanGame::addVertexNormals(vector<VertexType> vertices) {
   for (unsigned int i = 0; i < vertices.size(); i += 3) {
      vec3 p1 = vertices[i].pos;
      vec3 p2 = vertices[i+1].pos;
      vec3 p3 = vertices[i+2].pos;

      vec3 normal = normalize(cross(p2 - p1, p3 - p1));

      // Add the same normal for all 3 vertices
      vertices[i].normal = normal;
      vertices[i+1].normal = normal;
      vertices[i+2].normal = normal;
   }

   return vertices;
}

template<class VertexType>
vector<VertexType> VulkanGame::addObjectIndex(unsigned int objIndex, vector<VertexType> vertices) {
   for (VertexType& vertex : vertices) {
      vertex.objIndex = objIndex;
   }

   return vertices;
}

template<class VertexType, class SSBOType>
void VulkanGame::centerObject(SceneObject<VertexType, SSBOType>& object) {
   vector<VertexType>& vertices = object.vertices;

   float min_x = vertices[0].pos.x;
   float max_x = vertices[0].pos.x;
   float min_y = vertices[0].pos.y;
   float max_y = vertices[0].pos.y;
   float min_z = vertices[0].pos.z;
   float max_z = vertices[0].pos.z;

   // start from the second point
   for (unsigned int i = 1; i < vertices.size(); i++) {
      vec3& pos = vertices[i].pos;

      if (min_x > pos.x) {
         min_x = pos.x;
      } else if (max_x < pos.x) {
         max_x = pos.x;
      }

      if (min_y > pos.y) {
         min_y = pos.y;
      } else if (max_y < pos.y) {
         max_y = pos.y;
      }

      if (min_z > pos.z) {
         min_z = pos.z;
      } else if (max_z < pos.z) {
         max_z = pos.z;
      }
   }

   vec3 center = vec3(min_x + max_x, min_y + max_y, min_z + max_z) / 2.0f;

   for (unsigned int i = 0; i < vertices.size(); i++) {
      vertices[i].pos -= center;
   }

   object.radius = std::max(max_x - center.x, max_y - center.y);
   object.radius = std::max(object.radius, max_z - center.z);

   object.center = vec3(0.0f, 0.0f, 0.0f);
}

#endif // _VULKAN_GAME_H
