#ifndef _VULKAN_GAME_H
#define _VULKAN_GAME_H

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>

#include "game-gui-sdl.hpp"
#include "graphics-pipeline_vulkan.hpp"

#include "vulkan-utils.hpp"

using namespace glm;

#ifdef NDEBUG
   const bool ENABLE_VALIDATION_LAYERS = false;
#else
   const bool ENABLE_VALIDATION_LAYERS = true;
#endif

struct ModelVertex {
   vec3 pos;
   vec3 color;
   vec2 texCoord;
};

struct OverlayVertex {
   vec3 pos;
   vec2 texCoord;
};

struct ShipVertex {
   vec3 pos;
   vec3 color;
};

struct UBO_MvpMat {
   alignas(16) mat4 model;
   alignas(16) mat4 view;
   alignas(16) mat4 proj;
};

class VulkanGame {
   public:
      VulkanGame(int maxFramesInFlight);
      ~VulkanGame();

      void run(int width, int height, unsigned char guiFlags);

   private:
      const int MAX_FRAMES_IN_FLIGHT;

      const float NEAR_CLIP = 0.1f;
      const float FAR_CLIP = 100.0f;
      const float FOV_ANGLE = 67.0f;

      vec3 cam_pos;

      UBO_MvpMat modelMvpMats;
      UBO_MvpMat shipMvpMats;

      GameGui* gui;

      GraphicsPipeline_Vulkan<ModelVertex> modelPipeline;
      GraphicsPipeline_Vulkan<OverlayVertex> overlayPipeline;
      GraphicsPipeline_Vulkan<ShipVertex> shipPipeline;

      SDL_version sdlVersion;
      SDL_Window* window = nullptr;
      SDL_Renderer* renderer = nullptr;

      SDL_Texture* uiOverlay = nullptr;

      VkInstance instance;
      VkDebugUtilsMessengerEXT debugMessenger;
      VkSurfaceKHR surface; // TODO: Change the variable name to vulkanSurface
      VkPhysicalDevice physicalDevice = VK_NULL_HANDLE;
      VkDevice device;

      VkQueue graphicsQueue;
      VkQueue presentQueue;

      VkSwapchainKHR swapChain;
      vector<VkImage> swapChainImages;
      VkFormat swapChainImageFormat;
      VkExtent2D swapChainExtent;
      vector<VkImageView> swapChainImageViews;
      vector<VkFramebuffer> swapChainFramebuffers;

      VkRenderPass renderPass;
      VkCommandPool commandPool;
      vector<VkCommandBuffer> commandBuffers;

      VulkanImage depthImage;

      VkSampler textureSampler;

      // These are currently to store the MVP matrix
      // I should figure out if it makes sense to use them for other uniforms in the future
      // If not, I should rename them to better indicate their purpose.
      vector<VkBuffer> uniformBuffers;
      vector<VkDeviceMemory> uniformBuffersMemory;

      vector<VkDescriptorBufferInfo> uniformBufferInfoList;

      vector<VkBuffer> uniformBuffers_shipPipeline;
      vector<VkDeviceMemory> uniformBuffersMemory_shipPipeline;

      vector<VkDescriptorBufferInfo> uniformBufferInfoList_shipPipeline;

      VulkanImage floorTextureImage;
      VkDescriptorImageInfo floorTextureImageDescriptor;

      VulkanImage sdlOverlayImage;
      VkDescriptorImageInfo sdlOverlayImageDescriptor;

      TTF_Font* font;
      SDL_Texture* fontSDLTexture;

      SDL_Texture* imageSDLTexture;

      vector<VkSemaphore> imageAvailableSemaphores;
      vector<VkSemaphore> renderFinishedSemaphores;
      vector<VkFence> inFlightFences;

      size_t currentFrame;

      bool framebufferResized;

      bool initWindow(int width, int height, unsigned char guiFlags);
      void initVulkan();
      void initGraphicsPipelines();
      void initMatrices();
      void mainLoop();
      void updateScene(uint32_t currentImage);
      void renderUI();
      void renderScene();
      void cleanup();

      void createVulkanInstance(const vector<const char*> &validationLayers);
      void setupDebugMessenger();
      void populateDebugMessengerCreateInfo(VkDebugUtilsMessengerCreateInfoEXT& createInfo);
      void createVulkanSurface();
      void pickPhysicalDevice(const vector<const char*>& deviceExtensions);
      bool isDeviceSuitable(VkPhysicalDevice physicalDevice, const vector<const char*>& deviceExtensions);
      void createLogicalDevice(
         const vector<const char*> validationLayers,
         const vector<const char*>& deviceExtensions);
      void createSwapChain();
      void createImageViews();
      void createRenderPass();
      VkFormat findDepthFormat();
      void createCommandPool();
      void createImageResources();

      void createTextureSampler();
      void createFramebuffers();
      void createCommandBuffers();
      void createSyncObjects();

      template<class UniformType>
      void createUniformBuffers(vector<VkBuffer>& buffers, vector<VkDeviceMemory>& buffersMemory,
         vector<VkDescriptorBufferInfo>& bufferInfoList);

      void recreateSwapChain();

      void cleanupSwapChain();

      static VKAPI_ATTR VkBool32 VKAPI_CALL debugCallback(
            VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity,
            VkDebugUtilsMessageTypeFlagsEXT messageType,
            const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData,
            void* pUserData);
};

template<class UniformType>
void VulkanGame::createUniformBuffers(vector<VkBuffer>& buffers, vector<VkDeviceMemory>& buffersMemory,
      vector<VkDescriptorBufferInfo>& bufferInfoList) {
   buffers.resize(swapChainImages.size());
   buffersMemory.resize(swapChainImages.size());
   bufferInfoList.resize(swapChainImages.size());

   for (size_t i = 0; i < swapChainImages.size(); i++) {
      VulkanUtils::createBuffer(device, physicalDevice, sizeof(UniformType), VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT,
         VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT,
         buffers[i], buffersMemory[i]);

      bufferInfoList[i].buffer = buffers[i];
      bufferInfoList[i].offset = 0;
      bufferInfoList[i].range = sizeof(UniformType);
   }
}

#endif // _VULKAN_GAME_H