#include "sdl-game.hpp"

#include <array>
#include <iostream>
#include <set>
#include <stdexcept>

#include <SDL2/SDL_vulkan.h>

#include "IMGUI/imgui_impl_sdl.h"

#include "logger.hpp"

using namespace std;

#define IMGUI_UNLIMITED_FRAME_RATE

static void check_imgui_vk_result(VkResult res) {
   if (res == VK_SUCCESS) {
      return;
   }

   ostringstream oss;
   oss << "[imgui] Vulkan error! VkResult is \"" << VulkanUtils::resultString(res) << "\"" << __LINE__;
   if (res < 0) {
         throw runtime_error("Fatal: " + oss.str());
   } else {
         cerr << oss.str();
   }
}

VKAPI_ATTR VkBool32 VKAPI_CALL VulkanGame::debugCallback(
   VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity,
   VkDebugUtilsMessageTypeFlagsEXT messageType,
   const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData,
   void* pUserData) {
   cerr << "validation layer: " << pCallbackData->pMessage << endl;

   return VK_FALSE;
}

VulkanGame::VulkanGame() {
   // TODO: Double-check whether initialization should happen in the header, where the variables are declared, or here
   // Also, decide whether to use this-> for all instance variables, or only when necessary

   this->debugMessenger = VK_NULL_HANDLE;

   this->gui = nullptr;
   this->window = nullptr;

   this->swapChainPresentMode = VK_PRESENT_MODE_MAX_ENUM_KHR;
   this->swapChainMinImageCount = 0;
}

VulkanGame::~VulkanGame() {
}

void VulkanGame::run(int width, int height, unsigned char guiFlags) {
   cout << "DEBUGGING IS " << (ENABLE_VALIDATION_LAYERS ? "ON" : "OFF") << endl;

   cout << "Vulkan Game" << endl;

   if (initUI(width, height, guiFlags) == RTWO_ERROR) {
      return;
   }

   initVulkan();

   // Create Descriptor Pool
   {
      VkDescriptorPoolSize pool_sizes[] =
      {
          { VK_DESCRIPTOR_TYPE_SAMPLER, 1000 },
          { VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER, 1000 },
          { VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE, 1000 },
          { VK_DESCRIPTOR_TYPE_STORAGE_IMAGE, 1000 },
          { VK_DESCRIPTOR_TYPE_UNIFORM_TEXEL_BUFFER, 1000 },
          { VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER, 1000 },
          { VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER, 1000 },
          { VK_DESCRIPTOR_TYPE_STORAGE_BUFFER, 1000 },
          { VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC, 1000 },
          { VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC, 1000 },
          { VK_DESCRIPTOR_TYPE_INPUT_ATTACHMENT, 1000 }
      };
      VkDescriptorPoolCreateInfo pool_info = {};
      pool_info.sType = VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO;
      pool_info.flags = VK_DESCRIPTOR_POOL_CREATE_FREE_DESCRIPTOR_SET_BIT;
      pool_info.maxSets = 1000 * IM_ARRAYSIZE(pool_sizes);
      pool_info.poolSizeCount = (uint32_t)IM_ARRAYSIZE(pool_sizes);
      pool_info.pPoolSizes = pool_sizes;

      VKUTIL_CHECK_RESULT(vkCreateDescriptorPool(device, &pool_info, nullptr, &descriptorPool),
         "failed to create descriptor pool");
   }

   // TODO: Do this in one place and save it instead of redoing it every time I need a queue family index
   QueueFamilyIndices indices = VulkanUtils::findQueueFamilies(physicalDevice, surface);

   // Setup Dear ImGui context
   IMGUI_CHECKVERSION();
   ImGui::CreateContext();
   ImGuiIO& io = ImGui::GetIO();
   //io.ConfigFlags |= ImGuiConfigFlags_NavEnableKeyboard;     // Enable Keyboard Controls
   //io.ConfigFlags |= ImGuiConfigFlags_NavEnableGamepad;      // Enable Gamepad Controls

   // Setup Dear ImGui style
   ImGui::StyleColorsDark();
   //ImGui::StyleColorsClassic();

   // Setup Platform/Renderer bindings
   ImGui_ImplSDL2_InitForVulkan(window);
   ImGui_ImplVulkan_InitInfo init_info = {};
   init_info.Instance = instance;
   init_info.PhysicalDevice = physicalDevice;
   init_info.Device = device;
   init_info.QueueFamily = indices.graphicsFamily.value();
   init_info.Queue = graphicsQueue;
   init_info.DescriptorPool = descriptorPool;
   init_info.Allocator = nullptr;
   init_info.MinImageCount = swapChainMinImageCount;
   init_info.ImageCount = swapChainImageCount;
   init_info.CheckVkResultFn = check_imgui_vk_result;
   ImGui_ImplVulkan_Init(&init_info, renderPass);

   // Load Fonts
   // - If no fonts are loaded, dear imgui will use the default font. You can also load multiple fonts and use ImGui::PushFont()/PopFont() to select them.
   // - AddFontFromFileTTF() will return the ImFont* so you can store it if you need to select the font among multiple.
   // - If the file cannot be loaded, the function will return NULL. Please handle those errors in your application (e.g. use an assertion, or display an error and quit).
   // - The fonts will be rasterized at a given size (w/ oversampling) and stored into a texture when calling ImFontAtlas::Build()/GetTexDataAsXXXX(), which ImGui_ImplXXXX_NewFrame below will call.
   // - Read 'docs/FONTS.md' for more instructions and details.
   // - Remember that in C/C++ if you want to include a backslash \ in a string literal you need to write a double backslash \\ !
   //io.Fonts->AddFontDefault();
   //io.Fonts->AddFontFromFileTTF("../../misc/fonts/Roboto-Medium.ttf", 16.0f);
   //io.Fonts->AddFontFromFileTTF("../../misc/fonts/Cousine-Regular.ttf", 15.0f);
   //io.Fonts->AddFontFromFileTTF("../../misc/fonts/DroidSans.ttf", 16.0f);
   //io.Fonts->AddFontFromFileTTF("../../misc/fonts/ProggyTiny.ttf", 10.0f);
   //ImFont* font = io.Fonts->AddFontFromFileTTF("c:\\Windows\\Fonts\\ArialUni.ttf", 18.0f, NULL, io.Fonts->GetGlyphRangesJapanese());
   //assert(font != NULL);

   // Upload Fonts
   {
      VkCommandBuffer commandBuffer = VulkanUtils::beginSingleTimeCommands(device, resourceCommandPool);

      ImGui_ImplVulkan_CreateFontsTexture(commandBuffer);

      VulkanUtils::endSingleTimeCommands(device, resourceCommandPool, commandBuffer, graphicsQueue);

      ImGui_ImplVulkan_DestroyFontUploadObjects();
   }

   // Our state
   bool show_demo_window = true;
   bool show_another_window = false;

   done = false;
   while (!done) {
      // Poll and handle events (inputs, window resize, etc.)
      // You can read the io.WantCaptureMouse, io.WantCaptureKeyboard flags to tell if dear imgui wants to use your inputs.
      // - When io.WantCaptureMouse is true, do not dispatch mouse input data to your main application.
      // - When io.WantCaptureKeyboard is true, do not dispatch keyboard input data to your main application.
      // Generally you may always pass all inputs to dear imgui, and hide them from your application based on those two flags.
      SDL_Event event;
      while (SDL_PollEvent(&event)) {
         ImGui_ImplSDL2_ProcessEvent(&event);
         if (event.type == SDL_QUIT)
            done = true;
         if (event.type == SDL_WINDOWEVENT && event.window.event == SDL_WINDOWEVENT_CLOSE && event.window.windowID == SDL_GetWindowID(window))
            done = true;
      }

      // Resize swap chain?
      if (shouldRecreateSwapChain) {
         int width, height;
         SDL_GetWindowSize(window, &width, &height);
         if (width > 0 && height > 0) {
            // TODO: This should be used if the min image count changes, presumably because a new surface was created
            // with a different image count or something like that. Maybe I want to add code to query for a new min image count
            // during swapchain recreation to take advantage of this
            ImGui_ImplVulkan_SetMinImageCount(swapChainMinImageCount);

            recreateSwapChain();

            imageIndex = 0;
            shouldRecreateSwapChain = false;
         }
      }

      // Start the Dear ImGui frame
      ImGui_ImplVulkan_NewFrame();
      ImGui_ImplSDL2_NewFrame(window);
      ImGui::NewFrame();

      // 1. Show the big demo window (Most of the sample code is in ImGui::ShowDemoWindow()! You can browse its code to learn more about Dear ImGui!).
      if (show_demo_window)
         ImGui::ShowDemoWindow(&show_demo_window);

      // 2. Show a simple window that we create ourselves. We use a Begin/End pair to created a named window.
      {
         static int counter = 0;

         ImGui::Begin("Hello, world!");                          // Create a window called "Hello, world!" and append into it.

         ImGui::Text("This is some useful text.");               // Display some text (you can use a format strings too)
         ImGui::Checkbox("Demo Window", &show_demo_window);      // Edit bools storing our window open/close state
         ImGui::Checkbox("Another Window", &show_another_window);

         if (ImGui::Button("Button"))                            // Buttons return true when clicked (most widgets return true when edited/activated)
            counter++;
         ImGui::SameLine();
         ImGui::Text("counter = %d", counter);

         ImGui::Text("Application average %.3f ms/frame (%.1f FPS)", 1000.0f / ImGui::GetIO().Framerate, ImGui::GetIO().Framerate);
         ImGui::End();
      }

      // 3. Show another simple window.
      if (show_another_window)
      {
         ImGui::Begin("Another Window", &show_another_window);   // Pass a pointer to our bool variable (the window will have a closing button that will clear the bool when clicked)
         ImGui::Text("Hello from another window!");
         if (ImGui::Button("Close Me"))
            show_another_window = false;
         ImGui::End();
      }

      // Rendering
      ImGui::Render();
      ImDrawData* draw_data = ImGui::GetDrawData();
      const bool is_minimized = (draw_data->DisplaySize.x <= 0.0f || draw_data->DisplaySize.y <= 0.0f);
      if (!is_minimized) {
         renderFrame(draw_data);
         presentFrame();
      }
   }

   cleanup();

   close_log();
}

bool VulkanGame::initUI(int width, int height, unsigned char guiFlags) {
   // TODO: Create a game-gui function to get the gui version and retrieve it that way

   SDL_VERSION(&sdlVersion); // This gets the compile-time version
   SDL_GetVersion(&sdlVersion); // This gets the runtime version

   cout << "SDL " <<
      to_string(sdlVersion.major) << "." <<
      to_string(sdlVersion.minor) << "." <<
      to_string(sdlVersion.patch) << endl;

   // TODO: Refactor the logger api to be more flexible,
   // esp. since gl_log() and gl_log_err() have issues printing anything besides strings
   restart_gl_log();
   gl_log("starting SDL\n%s.%s.%s",
      to_string(sdlVersion.major).c_str(),
      to_string(sdlVersion.minor).c_str(),
      to_string(sdlVersion.patch).c_str());

   // TODO: Use open_Log() and related functions instead of gl_log ones
   // TODO: In addition, delete the gl_log functions
   open_log();
   get_log() << "starting SDL" << endl;
   get_log() <<
      (int)sdlVersion.major << "." <<
      (int)sdlVersion.minor << "." <<
      (int)sdlVersion.patch << endl;

   // TODO: Put all fonts, textures, and images in the assets folder
   gui = new GameGui_SDL();

   if (gui->init() == RTWO_ERROR) {
      // TODO: Also print these sorts of errors to the log
      cout << "UI library could not be initialized!" << endl;
      cout << gui->getError() << endl;
      return RTWO_ERROR;
   }

   window = (SDL_Window*)gui->createWindow("Vulkan Game", width, height, guiFlags & GUI_FLAGS_WINDOW_FULLSCREEN);
   if (window == nullptr) {
      cout << "Window could not be created!" << endl;
      cout << gui->getError() << endl;
      return RTWO_ERROR;
   }

   cout << "Target window size: (" << width << ", " << height << ")" << endl;
   cout << "Actual window size: (" << gui->getWindowWidth() << ", " << gui->getWindowHeight() << ")" << endl;

   return RTWO_SUCCESS;
}

void VulkanGame::initVulkan() {
   const vector<const char*> validationLayers = {
      "VK_LAYER_KHRONOS_validation"
   };
   const vector<const char*> deviceExtensions = {
      VK_KHR_SWAPCHAIN_EXTENSION_NAME
   };

   createVulkanInstance(validationLayers);
   setupDebugMessenger();
   createVulkanSurface();
   pickPhysicalDevice(deviceExtensions);
   createLogicalDevice(validationLayers, deviceExtensions);
   chooseSwapChainProperties();
   createSwapChain();
   createImageViews();
   createRenderPass();
   createResourceCommandPool();

   createCommandPools();

   VulkanUtils::createDepthImage(device, physicalDevice, resourceCommandPool, findDepthFormat(), swapChainExtent,
      depthImage, graphicsQueue);

   createFramebuffers();

   // TODO: Initialize pipelines here

   createCommandBuffers();

   createSyncObjects();
}

void VulkanGame::cleanup() {
   // FIXME: We could wait on the Queue if we had the queue in wd-> (otherwise VulkanH functions can't use globals)
    //vkQueueWaitIdle(g_Queue);
   if (vkDeviceWaitIdle(device) != VK_SUCCESS) {
      throw runtime_error("failed to wait for device!");
   }

   ImGui_ImplVulkan_Shutdown();
   ImGui_ImplSDL2_Shutdown();
   ImGui::DestroyContext();

   cleanupSwapChain();

   // this would actually be destroyed in the pipeline class
   vkDestroyDescriptorPool(device, descriptorPool, nullptr);

   vkDestroyCommandPool(device, resourceCommandPool, nullptr);

   vkDestroyDevice(device, nullptr);
   vkDestroySurfaceKHR(instance, surface, nullptr);

   if (ENABLE_VALIDATION_LAYERS) {
      VulkanUtils::destroyDebugUtilsMessengerEXT(instance, debugMessenger, nullptr);
   }

   vkDestroyInstance(instance, nullptr);

   gui->destroyWindow();
   gui->shutdown();
   delete gui;
}

void VulkanGame::createVulkanInstance(const vector<const char*>& validationLayers) {
   if (ENABLE_VALIDATION_LAYERS && !VulkanUtils::checkValidationLayerSupport(validationLayers)) {
      throw runtime_error("validation layers requested, but not available!");
   }

   VkApplicationInfo appInfo = {};
   appInfo.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
   appInfo.pApplicationName = "Vulkan Game";
   appInfo.applicationVersion = VK_MAKE_VERSION(1, 0, 0);
   appInfo.pEngineName = "No Engine";
   appInfo.engineVersion = VK_MAKE_VERSION(1, 0, 0);
   appInfo.apiVersion = VK_API_VERSION_1_0;

   VkInstanceCreateInfo createInfo = {};
   createInfo.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
   createInfo.pApplicationInfo = &appInfo;

   vector<const char*> extensions = gui->getRequiredExtensions();
   if (ENABLE_VALIDATION_LAYERS) {
      extensions.push_back(VK_EXT_DEBUG_UTILS_EXTENSION_NAME);
   }

   createInfo.enabledExtensionCount = static_cast<uint32_t>(extensions.size());
   createInfo.ppEnabledExtensionNames = extensions.data();

   cout << endl << "Extensions:" << endl;
   for (const char* extensionName : extensions) {
      cout << extensionName << endl;
   }
   cout << endl;

   VkDebugUtilsMessengerCreateInfoEXT debugCreateInfo;
   if (ENABLE_VALIDATION_LAYERS) {
      createInfo.enabledLayerCount = static_cast<uint32_t>(validationLayers.size());
      createInfo.ppEnabledLayerNames = validationLayers.data();

      populateDebugMessengerCreateInfo(debugCreateInfo);
      createInfo.pNext = &debugCreateInfo;
   }
   else {
      createInfo.enabledLayerCount = 0;

      createInfo.pNext = nullptr;
   }

   if (vkCreateInstance(&createInfo, nullptr, &instance) != VK_SUCCESS) {
      throw runtime_error("failed to create instance!");
   }
}

void VulkanGame::setupDebugMessenger() {
   if (!ENABLE_VALIDATION_LAYERS) {
      return;
   }

   VkDebugUtilsMessengerCreateInfoEXT createInfo;
   populateDebugMessengerCreateInfo(createInfo);

   if (VulkanUtils::createDebugUtilsMessengerEXT(instance, &createInfo, nullptr, &debugMessenger) != VK_SUCCESS) {
      throw runtime_error("failed to set up debug messenger!");
   }
}

void VulkanGame::populateDebugMessengerCreateInfo(VkDebugUtilsMessengerCreateInfoEXT& createInfo) {
   createInfo = {};
   createInfo.sType = VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT;
   createInfo.messageSeverity = VK_DEBUG_UTILS_MESSAGE_SEVERITY_VERBOSE_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_SEVERITY_WARNING_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_SEVERITY_ERROR_BIT_EXT;
   createInfo.messageType = VK_DEBUG_UTILS_MESSAGE_TYPE_GENERAL_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_TYPE_VALIDATION_BIT_EXT | VK_DEBUG_UTILS_MESSAGE_TYPE_PERFORMANCE_BIT_EXT;
   createInfo.pfnUserCallback = debugCallback;
}

void VulkanGame::createVulkanSurface() {
   if (gui->createVulkanSurface(instance, &surface) == RTWO_ERROR) {
      throw runtime_error("failed to create window surface!");
   }
}

void VulkanGame::pickPhysicalDevice(const vector<const char*>& deviceExtensions) {
   uint32_t deviceCount = 0;
   // TODO: Check VkResult
   vkEnumeratePhysicalDevices(instance, &deviceCount, nullptr);

   if (deviceCount == 0) {
      throw runtime_error("failed to find GPUs with Vulkan support!");
   }

   vector<VkPhysicalDevice> devices(deviceCount);
   // TODO: Check VkResult
   vkEnumeratePhysicalDevices(instance, &deviceCount, devices.data());

   cout << endl << "Graphics cards:" << endl;
   for (const VkPhysicalDevice& device : devices) {
      if (isDeviceSuitable(device, deviceExtensions)) {
         physicalDevice = device;
         break;
      }
   }
   cout << endl;

   if (physicalDevice == VK_NULL_HANDLE) {
      throw runtime_error("failed to find a suitable GPU!");
   }
}

bool VulkanGame::isDeviceSuitable(VkPhysicalDevice physicalDevice, const vector<const char*>& deviceExtensions) {
   VkPhysicalDeviceProperties deviceProperties;
   vkGetPhysicalDeviceProperties(physicalDevice, &deviceProperties);

   cout << "Device: " << deviceProperties.deviceName << endl;

   QueueFamilyIndices indices = VulkanUtils::findQueueFamilies(physicalDevice, surface);
   bool extensionsSupported = VulkanUtils::checkDeviceExtensionSupport(physicalDevice, deviceExtensions);
   bool swapChainAdequate = false;

   if (extensionsSupported) {
      vector<VkSurfaceFormatKHR> formats = VulkanUtils::querySwapChainFormats(physicalDevice, surface);
      vector<VkPresentModeKHR> presentModes = VulkanUtils::querySwapChainPresentModes(physicalDevice, surface);

      swapChainAdequate = !formats.empty() && !presentModes.empty();
   }

   VkPhysicalDeviceFeatures supportedFeatures;
   vkGetPhysicalDeviceFeatures(physicalDevice, &supportedFeatures);

   return indices.isComplete() && extensionsSupported && swapChainAdequate && supportedFeatures.samplerAnisotropy;
}

void VulkanGame::createLogicalDevice(const vector<const char*>& validationLayers,
   const vector<const char*>& deviceExtensions) {
   QueueFamilyIndices indices = VulkanUtils::findQueueFamilies(physicalDevice, surface);

   if (!indices.isComplete()) {
      throw runtime_error("failed to find required queue families!");
   }

   // TODO: Using separate graphics and present queues currently works, but I should verify that I'm
   // using them correctly to get the most benefit out of separate queues

   vector<VkDeviceQueueCreateInfo> queueCreateInfoList;
   set<uint32_t> uniqueQueueFamilies = { indices.graphicsFamily.value(), indices.presentFamily.value() };

   float queuePriority = 1.0f;
   for (uint32_t queueFamily : uniqueQueueFamilies) {
      VkDeviceQueueCreateInfo queueCreateInfo = {};
      queueCreateInfo.sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
      queueCreateInfo.queueCount = 1;
      queueCreateInfo.queueFamilyIndex = queueFamily;
      queueCreateInfo.pQueuePriorities = &queuePriority;

      queueCreateInfoList.push_back(queueCreateInfo);
   }

   VkPhysicalDeviceFeatures deviceFeatures = {};
   deviceFeatures.samplerAnisotropy = VK_TRUE;

   VkDeviceCreateInfo createInfo = {};
   createInfo.sType = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO;

   createInfo.queueCreateInfoCount = static_cast<uint32_t>(queueCreateInfoList.size());
   createInfo.pQueueCreateInfos = queueCreateInfoList.data();

   createInfo.pEnabledFeatures = &deviceFeatures;

   createInfo.enabledExtensionCount = static_cast<uint32_t>(deviceExtensions.size());
   createInfo.ppEnabledExtensionNames = deviceExtensions.data();

   // These fields are ignored  by up-to-date Vulkan implementations,
   // but it's a good idea to set them for backwards compatibility
   if (ENABLE_VALIDATION_LAYERS) {
      createInfo.enabledLayerCount = static_cast<uint32_t>(validationLayers.size());
      createInfo.ppEnabledLayerNames = validationLayers.data();
   }
   else {
      createInfo.enabledLayerCount = 0;
   }

   if (vkCreateDevice(physicalDevice, &createInfo, nullptr, &device) != VK_SUCCESS) {
      throw runtime_error("failed to create logical device!");
   }

   vkGetDeviceQueue(device, indices.graphicsFamily.value(), 0, &graphicsQueue);
   vkGetDeviceQueue(device, indices.presentFamily.value(), 0, &presentQueue);
}

void VulkanGame::chooseSwapChainProperties() {
   vector<VkSurfaceFormatKHR> availableFormats = VulkanUtils::querySwapChainFormats(physicalDevice, surface);
   vector<VkPresentModeKHR> availablePresentModes = VulkanUtils::querySwapChainPresentModes(physicalDevice, surface);

   // Per Spec Format and View Format are expected to be the same unless VK_IMAGE_CREATE_MUTABLE_BIT was set at image creation
   // Assuming that the default behavior is without setting this bit, there is no need for separate Swapchain image and image view format
   // Additionally several new color spaces were introduced with Vulkan Spec v1.0.40,
   // hence we must make sure that a format with the mostly available color space, VK_COLOR_SPACE_SRGB_NONLINEAR_KHR, is found and used.
   swapChainSurfaceFormat = VulkanUtils::chooseSwapSurfaceFormat(availableFormats,
      { VK_FORMAT_B8G8R8A8_UNORM, VK_FORMAT_R8G8B8A8_UNORM, VK_FORMAT_B8G8R8_UNORM, VK_FORMAT_R8G8B8_UNORM },
      VK_COLOR_SPACE_SRGB_NONLINEAR_KHR);

#ifdef IMGUI_UNLIMITED_FRAME_RATE
   vector<VkPresentModeKHR> presentModes{
      VK_PRESENT_MODE_MAILBOX_KHR, VK_PRESENT_MODE_IMMEDIATE_KHR, VK_PRESENT_MODE_FIFO_KHR
   };
#else
   vector<VkPresentModeKHR> presentModes{ VK_PRESENT_MODE_FIFO_KHR };
#endif

   swapChainPresentMode = VulkanUtils::chooseSwapPresentMode(availablePresentModes, presentModes);

   cout << "[vulkan] Selected PresentMode = " << swapChainPresentMode << endl;

   VkSurfaceCapabilitiesKHR capabilities = VulkanUtils::querySwapChainCapabilities(physicalDevice, surface);

   // If min image count was not specified, request different count of images dependent on selected present mode
   if (swapChainMinImageCount == 0) {
      if (swapChainPresentMode == VK_PRESENT_MODE_MAILBOX_KHR) {
         swapChainMinImageCount = 3;
      }
      else if (swapChainPresentMode == VK_PRESENT_MODE_FIFO_KHR || swapChainPresentMode == VK_PRESENT_MODE_FIFO_RELAXED_KHR) {
         swapChainMinImageCount = 2;
      }
      else if (swapChainPresentMode == VK_PRESENT_MODE_IMMEDIATE_KHR) {
         swapChainMinImageCount = 1;
      }
      else {
         throw runtime_error("unexpected present mode!");
      }
   }

   if (swapChainMinImageCount < capabilities.minImageCount) {
      swapChainMinImageCount = capabilities.minImageCount;
   }
   else if (capabilities.maxImageCount != 0 && swapChainMinImageCount > capabilities.maxImageCount) {
      swapChainMinImageCount = capabilities.maxImageCount;
   }
}

void VulkanGame::createSwapChain() {
   VkSurfaceCapabilitiesKHR capabilities = VulkanUtils::querySwapChainCapabilities(physicalDevice, surface);

   swapChainExtent = VulkanUtils::chooseSwapExtent(capabilities, gui->getWindowWidth(), gui->getWindowHeight());

   VkSwapchainCreateInfoKHR createInfo = {};
   createInfo.sType = VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR;
   createInfo.surface = surface;
   createInfo.minImageCount = swapChainMinImageCount;
   createInfo.imageFormat = swapChainSurfaceFormat.format;
   createInfo.imageColorSpace = swapChainSurfaceFormat.colorSpace;
   createInfo.imageExtent = swapChainExtent;
   createInfo.imageArrayLayers = 1;
   createInfo.imageUsage = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT;

   // TODO: Maybe save this result so I don't have to recalculate it every time
   QueueFamilyIndices indices = VulkanUtils::findQueueFamilies(physicalDevice, surface);
   uint32_t queueFamilyIndices[] = { indices.graphicsFamily.value(), indices.presentFamily.value() };

   if (indices.graphicsFamily != indices.presentFamily) {
      createInfo.imageSharingMode = VK_SHARING_MODE_CONCURRENT;
      createInfo.queueFamilyIndexCount = 2;
      createInfo.pQueueFamilyIndices = queueFamilyIndices;
   }
   else {
      createInfo.imageSharingMode = VK_SHARING_MODE_EXCLUSIVE;
      createInfo.queueFamilyIndexCount = 0;
      createInfo.pQueueFamilyIndices = nullptr;
   }

   createInfo.preTransform = capabilities.currentTransform;
   createInfo.compositeAlpha = VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR;
   createInfo.presentMode = swapChainPresentMode;
   createInfo.clipped = VK_TRUE;
   createInfo.oldSwapchain = VK_NULL_HANDLE;

   if (vkCreateSwapchainKHR(device, &createInfo, nullptr, &swapChain) != VK_SUCCESS) {
      throw runtime_error("failed to create swap chain!");
   }

   if (vkGetSwapchainImagesKHR(device, swapChain, &swapChainImageCount, nullptr) != VK_SUCCESS) {
      throw runtime_error("failed to get swap chain image count!");
   }

   swapChainImages.resize(swapChainImageCount);
   if (vkGetSwapchainImagesKHR(device, swapChain, &swapChainImageCount, swapChainImages.data()) != VK_SUCCESS) {
      throw runtime_error("failed to get swap chain images!");
   }
}

void VulkanGame::createImageViews() {
   swapChainImageViews.resize(swapChainImageCount);

   for (uint32_t i = 0; i < swapChainImageViews.size(); i++) {
      swapChainImageViews[i] = VulkanUtils::createImageView(device, swapChainImages[i], swapChainSurfaceFormat.format,
         VK_IMAGE_ASPECT_COLOR_BIT);
   }
}

void VulkanGame::createRenderPass() {
   VkAttachmentDescription colorAttachment = {};
   colorAttachment.format = swapChainSurfaceFormat.format;
   colorAttachment.samples = VK_SAMPLE_COUNT_1_BIT;
   colorAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR; // Set to VK_ATTACHMENT_LOAD_OP_DONT_CARE to disable clearing
   colorAttachment.storeOp = VK_ATTACHMENT_STORE_OP_STORE;
   colorAttachment.stencilLoadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
   colorAttachment.stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
   colorAttachment.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
   colorAttachment.finalLayout = VK_IMAGE_LAYOUT_PRESENT_SRC_KHR;

   VkAttachmentReference colorAttachmentRef = {};
   colorAttachmentRef.attachment = 0;
   colorAttachmentRef.layout = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;

   VkAttachmentDescription depthAttachment = {};
   depthAttachment.format = findDepthFormat();
   depthAttachment.samples = VK_SAMPLE_COUNT_1_BIT;
   depthAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
   depthAttachment.storeOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
   depthAttachment.stencilLoadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
   depthAttachment.stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
   depthAttachment.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
   depthAttachment.finalLayout = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;

   VkAttachmentReference depthAttachmentRef = {};
   depthAttachmentRef.attachment = 1;
   depthAttachmentRef.layout = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;

   VkSubpassDescription subpass = {};
   subpass.pipelineBindPoint = VK_PIPELINE_BIND_POINT_GRAPHICS;
   subpass.colorAttachmentCount = 1;
   subpass.pColorAttachments = &colorAttachmentRef;
   //subpass.pDepthStencilAttachment = &depthAttachmentRef;

   VkSubpassDependency dependency = {};
   dependency.srcSubpass = VK_SUBPASS_EXTERNAL;
   dependency.dstSubpass = 0;
   dependency.srcStageMask = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
   dependency.srcAccessMask = 0;
   dependency.dstStageMask = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
   dependency.dstAccessMask = VK_ACCESS_COLOR_ATTACHMENT_READ_BIT | VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;

   array<VkAttachmentDescription, 2> attachments = { colorAttachment, depthAttachment };
   VkRenderPassCreateInfo renderPassInfo = {};
   renderPassInfo.sType = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO;
   renderPassInfo.attachmentCount = static_cast<uint32_t>(attachments.size());
   renderPassInfo.pAttachments = attachments.data();
   renderPassInfo.subpassCount = 1;
   renderPassInfo.pSubpasses = &subpass;
   renderPassInfo.dependencyCount = 1;
   renderPassInfo.pDependencies = &dependency;

   if (vkCreateRenderPass(device, &renderPassInfo, nullptr, &renderPass) != VK_SUCCESS) {
      throw runtime_error("failed to create render pass!");
   }

   // We do not create a pipeline by default as this is also used by examples' main.cpp,
   // but secondary viewport in multi-viewport mode may want to create one with:
   //ImGui_ImplVulkan_CreatePipeline(device, g_Allocator, VK_NULL_HANDLE, g_MainWindowData.RenderPass, VK_SAMPLE_COUNT_1_BIT, &g_MainWindowData.Pipeline);
}

VkFormat VulkanGame::findDepthFormat() {
   return VulkanUtils::findSupportedFormat(
      physicalDevice,
      { VK_FORMAT_D32_SFLOAT, VK_FORMAT_D32_SFLOAT_S8_UINT, VK_FORMAT_D24_UNORM_S8_UINT },
      VK_IMAGE_TILING_OPTIMAL,
      VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT
   );
}

void VulkanGame::createResourceCommandPool() {
   QueueFamilyIndices indices = VulkanUtils::findQueueFamilies(physicalDevice, surface);

   VkCommandPoolCreateInfo poolInfo = {};
   poolInfo.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
   poolInfo.queueFamilyIndex = indices.graphicsFamily.value();
   poolInfo.flags = 0;

   if (vkCreateCommandPool(device, &poolInfo, nullptr, &resourceCommandPool) != VK_SUCCESS) {
      throw runtime_error("failed to create resource command pool!");
   }
}

void VulkanGame::createCommandPools() {
   commandPools.resize(swapChainImageCount);

   QueueFamilyIndices indices = VulkanUtils::findQueueFamilies(physicalDevice, surface);

   for (size_t i = 0; i < swapChainImageCount; i++) {
      VkCommandPoolCreateInfo poolInfo = {};
      poolInfo.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
      poolInfo.queueFamilyIndex = indices.graphicsFamily.value();
      poolInfo.flags = VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT;
      if (vkCreateCommandPool(device, &poolInfo, nullptr, &commandPools[i]) != VK_SUCCESS) {
         throw runtime_error("failed to create graphics command pool!");
      }
   }
}

void VulkanGame::createFramebuffers() {
   swapChainFramebuffers.resize(swapChainImageCount);

   VkFramebufferCreateInfo framebufferInfo = {};
   framebufferInfo.sType = VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO;
   framebufferInfo.renderPass = renderPass;
   framebufferInfo.width = swapChainExtent.width;
   framebufferInfo.height = swapChainExtent.height;
   framebufferInfo.layers = 1;

   for (size_t i = 0; i < swapChainImageCount; i++) {
      array<VkImageView, 2> attachments = {
         swapChainImageViews[i],
         depthImage.imageView
      };

      framebufferInfo.attachmentCount = static_cast<uint32_t>(attachments.size());
      framebufferInfo.pAttachments = attachments.data();

      if (vkCreateFramebuffer(device, &framebufferInfo, nullptr, &swapChainFramebuffers[i]) != VK_SUCCESS) {
         throw runtime_error("failed to create framebuffer!");
      }
   }
}

void VulkanGame::createCommandBuffers() {
   commandBuffers.resize(swapChainImageCount);

   for (size_t i = 0; i < swapChainImageCount; i++) {
      VkCommandBufferAllocateInfo allocInfo = {};
      allocInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
      allocInfo.commandPool = commandPools[i];
      allocInfo.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
      allocInfo.commandBufferCount = 1;

      if (vkAllocateCommandBuffers(device, &allocInfo, &commandBuffers[i]) != VK_SUCCESS) {
         throw runtime_error("failed to allocate command buffers!");
      }
   }
}

void VulkanGame::createSyncObjects() {
   imageAcquiredSemaphores.resize(swapChainImageCount);
   renderCompleteSemaphores.resize(swapChainImageCount);
   inFlightFences.resize(swapChainImageCount);

   VkSemaphoreCreateInfo semaphoreInfo = {};
   semaphoreInfo.sType = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO;

   VkFenceCreateInfo fenceInfo = {};
   fenceInfo.sType = VK_STRUCTURE_TYPE_FENCE_CREATE_INFO;
   fenceInfo.flags = VK_FENCE_CREATE_SIGNALED_BIT;

   for (size_t i = 0; i < swapChainImageCount; i++) {
      if (vkCreateSemaphore(device, &semaphoreInfo, nullptr, &imageAcquiredSemaphores[i]) != VK_SUCCESS) {
         throw runtime_error("failed to create image acquired sempahore for a frame!");
      }

      if (vkCreateSemaphore(device, &semaphoreInfo, nullptr, &renderCompleteSemaphores[i]) != VK_SUCCESS) {
         throw runtime_error("failed to create render complete sempahore for a frame!");
      }

      if (vkCreateFence(device, &fenceInfo, nullptr, &inFlightFences[i]) != VK_SUCCESS) {
         throw runtime_error("failed to create fence for a frame!");
      }
   }
}

void VulkanGame::renderFrame(ImDrawData* draw_data) {
   VkResult result = vkAcquireNextImageKHR(device, swapChain, numeric_limits<uint64_t>::max(),
      imageAcquiredSemaphores[currentFrame], VK_NULL_HANDLE, &imageIndex);

   if (result == VK_ERROR_OUT_OF_DATE_KHR) {
      shouldRecreateSwapChain = true;
      return;
   } else {
      VKUTIL_CHECK_RESULT(result, "failed to acquire swap chain image!");
   }

   VKUTIL_CHECK_RESULT(vkWaitForFences(device, 1, &inFlightFences[imageIndex], VK_TRUE, numeric_limits<uint64_t>::max()),
      "failed waiting for fence!");

   VKUTIL_CHECK_RESULT(vkResetFences(device, 1, &inFlightFences[imageIndex]),
      "failed to reset fence!");

   // START OF NEW CODE
   // I don't have analogous code in vulkan-game right now because I record command buffers once
   // before the render loop ever starts. I should change this

   VKUTIL_CHECK_RESULT(vkResetCommandPool(device, commandPools[imageIndex], 0),
      "failed to reset command pool!");

   VkCommandBufferBeginInfo info = {};
   info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
   info.flags |= VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT;

   VKUTIL_CHECK_RESULT(vkBeginCommandBuffer(commandBuffers[imageIndex], &info),
      "failed to begin recording command buffer!");

   VkRenderPassBeginInfo renderPassInfo = {};
   renderPassInfo.sType = VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO;
   renderPassInfo.renderPass = renderPass;
   renderPassInfo.framebuffer = swapChainFramebuffers[imageIndex];
   renderPassInfo.renderArea.extent = swapChainExtent;

   array<VkClearValue, 2> clearValues = {};
   clearValues[0].color = { { 0.45f, 0.55f, 0.60f, 1.00f } };
   clearValues[1].depthStencil = { 1.0f, 0 };

   renderPassInfo.clearValueCount = static_cast<uint32_t>(clearValues.size());
   renderPassInfo.pClearValues = clearValues.data();

   vkCmdBeginRenderPass(commandBuffers[imageIndex], &renderPassInfo, VK_SUBPASS_CONTENTS_INLINE);

   // Record dear imgui primitives into command buffer
   ImGui_ImplVulkan_RenderDrawData(draw_data, commandBuffers[imageIndex]);

   // Submit command buffer
   vkCmdEndRenderPass(commandBuffers[imageIndex]);

   VKUTIL_CHECK_RESULT(vkEndCommandBuffer(commandBuffers[imageIndex]),
      "failed to record command buffer!");

   // END OF NEW CODE

   VkSemaphore waitSemaphores[] = { imageAcquiredSemaphores[currentFrame] };
   VkPipelineStageFlags wait_stage = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
   VkSemaphore signalSemaphores[] = { renderCompleteSemaphores[currentFrame] };

   VkSubmitInfo submitInfo = {};
   submitInfo.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
   submitInfo.waitSemaphoreCount = 1;
   submitInfo.pWaitSemaphores = waitSemaphores;
   submitInfo.pWaitDstStageMask = &wait_stage;
   submitInfo.commandBufferCount = 1;
   submitInfo.pCommandBuffers = &commandBuffers[imageIndex];
   submitInfo.signalSemaphoreCount = 1;
   submitInfo.pSignalSemaphores = signalSemaphores;

   VKUTIL_CHECK_RESULT(vkQueueSubmit(graphicsQueue, 1, &submitInfo, inFlightFences[imageIndex]),
      "failed to submit draw command buffer!");
}

void VulkanGame::presentFrame() {
   if (shouldRecreateSwapChain) {
      return;
   }

   VkSemaphore signalSemaphores[] = { renderCompleteSemaphores[currentFrame] };

   VkPresentInfoKHR presentInfo = {};
   presentInfo.sType = VK_STRUCTURE_TYPE_PRESENT_INFO_KHR;
   presentInfo.waitSemaphoreCount = 1;
   presentInfo.pWaitSemaphores = signalSemaphores;
   presentInfo.swapchainCount = 1;
   presentInfo.pSwapchains = &swapChain;
   presentInfo.pImageIndices = &imageIndex;
   presentInfo.pResults = nullptr;

   VkResult result = vkQueuePresentKHR(presentQueue, &presentInfo);

   // In vulkan-game, I also handle VK_SUBOPTIMAL_KHR and framebufferResized. g_SwapChainRebuild is kind of similar
   // to framebufferResized, but not quite the same
   if (result == VK_ERROR_OUT_OF_DATE_KHR) {
      shouldRecreateSwapChain = true;
      return;
   } else if (result != VK_SUCCESS) {
      throw runtime_error("failed to present swap chain image!");
   }

   currentFrame = (currentFrame + 1) % swapChainImageCount;
}

void VulkanGame::recreateSwapChain() {
   if (vkDeviceWaitIdle(device) != VK_SUCCESS) {
      throw runtime_error("failed to wait for device!");
   }

   cleanupSwapChain();

   createSwapChain();
   createImageViews();
   createRenderPass();

   createCommandPools();

   // The depth buffer does need to be recreated with the swap chain since its dimensions depend on the window size
   // and resizing the window is a common reason to recreate the swapchain
   VulkanUtils::createDepthImage(device, physicalDevice, resourceCommandPool, findDepthFormat(), swapChainExtent,
      depthImage, graphicsQueue);

   createFramebuffers();

   // TODO: Update pipelines here

   createCommandBuffers();

   createSyncObjects();
}

void VulkanGame::cleanupSwapChain() {
   VulkanUtils::destroyVulkanImage(device, depthImage);

   for (VkFramebuffer framebuffer : swapChainFramebuffers) {
      vkDestroyFramebuffer(device, framebuffer, nullptr);
   }

   for (uint32_t i = 0; i < swapChainImageCount; i++) {
      vkFreeCommandBuffers(device, commandPools[i], 1, &commandBuffers[i]);
      vkDestroyCommandPool(device, commandPools[i], nullptr);
   }

   for (uint32_t i = 0; i < swapChainImageCount; i++) {
      vkDestroySemaphore(device, imageAcquiredSemaphores[i], nullptr);
      vkDestroySemaphore(device, renderCompleteSemaphores[i], nullptr);
      vkDestroyFence(device, inFlightFences[i], nullptr);
   }

   vkDestroyRenderPass(device, renderPass, nullptr);

   for (VkImageView imageView : swapChainImageViews) {
      vkDestroyImageView(device, imageView, nullptr);
   }

   vkDestroySwapchainKHR(device, swapChain, nullptr);
}
